from pathlib import Path
from importlib import resources
from typing import Union
import logging
from fpdf import FPDF
from fpdf.enums import CellBordersLayout, Align


from qrcode import QRCode
from qrcode.constants import ERROR_CORRECT_L

from table_maker.helper import scale_img, get_images, to_cur

log = logging.getLogger(__name__)


def make_ad(
    p: Path,
    plaza_txt_mtime: float,
    id: str,
    title: str,
    short_url: str,
    price: str,
) -> Union[FPDF, None]:
    dest_file = p.joinpath(f"Publico/Exposee/{id}-anzeige-48x60.pdf")
    ppm = 5

    if dest_file.exists() and dest_file.stat().st_mtime > plaza_txt_mtime:
        return None

    main_image = get_images(p.joinpath("Publico/Imagen"))

    if main_image is None or len(main_image) < 1:
        return None
    main_image = main_image[0]

    price = to_cur(price) + "* EUR"

    pdf = FPDF(orientation="P", unit="mm", format=(65, 48))
    # Different styles of the same font family.
    with resources.as_file(
        resources.files("table_maker").joinpath("assets/FreeSans.ttf")
    ) as fp:
        pdf.add_font(
            "free-sans",
            style="",
            fname=fp,
        )
    with resources.as_file(
        resources.files("table_maker").joinpath("assets/FreeSansBold.ttf")
    ) as fp:
        pdf.add_font(
            "free-sans",
            style="B",
            fname=fp,
        )
    with resources.as_file(
        resources.files("table_maker").joinpath("assets/FreeSansOblique.ttf")
    ) as fp:
        pdf.add_font(
            "free-sans",
            style="I",
            fname=fp,
        )
    with resources.as_file(
        resources.files("table_maker").joinpath("assets/FreeSansBoldOblique.ttf")
    ) as fp:
        pdf.add_font(
            "free-sans",
            style="BI",
            fname=fp,
        )

    pdf.add_page()
    pdf.set_margin(0)

    pdf.set_font("free-sans", style="B", size=8)
    pdf.set_text_color(255, 255, 255)
    pdf.set_fill_color(119, 2, 21)
    with pdf.table(
        first_row_as_headings=False,
        text_align="C",
        line_height=4,
        borders_layout="none",
    ) as table:
        table.row().cell(title + " - La Palma")

    img = scale_img(main_image, ideal_width=round(pdf.epw * ppm), ideal_height=27 * ppm)
    if img is None:
        return None
    pdf.image(img, w=pdf.epw, h=27)

    qr = QRCode(version=1, error_correction=ERROR_CORRECT_L, box_size=10, border=2)
    qr.add_data(short_url)
    qr.make(fit=True)
    img_qr = qr.make_image(fill_color="black", back_color="white").get_image()

    pdf.image(
        img_qr,
        x=0,
        y=pdf.y - qr.box_size,
        w=qr.box_size,
        h=qr.box_size,
    )

    pdf.set_font("free-sans", style="B", size=8)
    pdf.cell(w=(pdf.epw - pdf.get_string_width(price) - 6))
    pdf.set_text_color(255)
    pdf.set_fill_color(119, 2, 21)
    pdf.cell(
        w=pdf.get_string_width(price) + 6,
        h=-4,
        text=price,
        fill=True,
        align=Align.C,
        #        new_x="LMARGIN",
    )

    with pdf.rotation(90, x=pdf.x - 3.5, y=pdf.y):
        pdf.set_text_color(0)
        pdf.set_font("free-sans", size=6, style="B")
        pdf.cell(None, None, "* zzgl. Courtage 3,21%", align=Align.R)

    pdf.set_font("free-sans", style="", size=9)
    pdf.set_text_color(119, 2, 21)
    pdf.set_fill_color(255)
    pdf.set_draw_color(119, 2, 21)
    with pdf.table(
        first_row_as_headings=False,
        text_align="L",
        line_height=4,
        borders_layout="none",
    ) as table:
        row = table.row()
        with resources.as_file(
            resources.files("table_maker") / "assets/logo.jpg"
        ) as fp:
            row.cell(
                img=fp.as_posix(),
                border=CellBordersLayout.BOTTOM | CellBordersLayout.LEFT,
                padding=(0.5, 0.5, 0.5, 0.5),
                align=Align.L,
            )
        row.cell(
            text="anfrage@plaza-\ninmobiliaria.de\n+34 603 101 281",
            border=CellBordersLayout.BOTTOM | CellBordersLayout.RIGHT,
            padding=(0.5, 0, 0.5, 7.5),
        )
    try:
        pdf.output(dest_file.as_posix())
    except Exception as e:
        log.exception(e, exc_info=True)


if __name__ == "__main__":
    make_ad(
        Path("/home/stephan/Plaza/Objekte/Norden/Garafia/ST680, 440.000 Horst"),
        plaza_txt_mtime=0,
        id="ST680",
        title="Historisches Haus mit Dachterasse und Meerblick in Garafia",
        short_url="https://plaza.short.gy/DNIwjH",
        price="440000",
    )
