import re
import configparser
import argparse
import os
import logging
from logging.handlers import TimedRotatingFileHandler
from pathlib import Path
from typing import Union, List, Dict, Any
from jinja2 import Environment, PackageLoader, select_autoescape
from table_maker import __version__
from table_maker.exposee import make_exposee
from table_maker.helper import scale_img, get_images
from table_maker.ad import make_ad
from table_maker.ftp import ftp_setup, ftp_upload

FTP_PASSWORD = "Jh6T3F2HZ5gEgvF5"
FTP_USER = "plazas_0"
FTP_HOST = "www174.your-server.de"

table_data: List[Dict[Any, Any]] = []
mymaps_csv = "WKT,Name,Beschreibung\n"
newest_plaza_txt_mtime: float = 0

log = logging.getLogger(__name__)


def main() -> None:
    global table_data
    global mymaps_csv
    global newest_plaza_txt_mtime

    parser = argparse.ArgumentParser()
    parser.add_argument(
        "-r",
        "--root",
        help="Start directory",
        default="/home/stephan/repos/table_maker/Objekte",
    )
    parser.add_argument(
        "--logfile",
        help="Path to logfile",
        # default="log/table_maker.log",
    )
    parser.add_argument(
        "-d",
        "--debug",
        help="Set debug level, from 0 to 1",
        type=int,
        default=0,
    )
    args = parser.parse_args()

    setup_logging(args.logfile, debug=args.debug)
    log.info(f"table_maker V{__version__} start")

    root = args.root
    ftp_setup(FTP_HOST, FTP_USER, FTP_PASSWORD)

    path_index_html = Path(root, "index.html")

    for subdir in ["Westen", "Norden", "Sueden", "Osten"]:
        p = Path(root, subdir)
        if p.exists():
            traverse(p, root)

    template = Environment(
        loader=PackageLoader("table_maker"),
        autoescape=select_autoescape(),
    ).get_template("table_maker.html")

    if path_index_html.exists():
        index_html_mtime = path_index_html.stat().st_mtime
    else:
        index_html_mtime = 0

    if newest_plaza_txt_mtime > index_html_mtime:
        try:
            with path_index_html.open("w") as f:
                f.write(template.render(data=table_data, version=__version__))
        except Exception as e:
            log.exception(e, exc_info=True)

        try:
            with Path(root, "mymaps.csv").open("w") as f:
                f.write(mymaps_csv)
        except Exception as e:
            log.exception(e, exc_info=True)

    ftp_upload(remote_filename="index.html", local_file_path=path_index_html)

    log.info("table_maker finished")


def traverse(p: Path, root: str):
    for current_dir in p.iterdir():
        if not current_dir.is_dir():
            continue

        ## check that directory was not moved the last 15 minutes
        #        lstat = current_dir.lstat()
        #        if lstat.st_atime > lstat.st_mtime and lstat.st_atime + (60 + 15) > time.time():
        #            continue

        object_id = re.match("[^\w]*(st|l|bg|w|gw|m)([0-9]+)", current_dir.name, re.I)  # pyright: ignore[reportInvalidStringEscapeSequence] # noqa: Q000
        if object_id:
            scan_object(
                current_dir,
                object_id.group(1),
                object_id.group(2),
                root,
            )
            continue

        # We have a directory that is no object, so dive inside
        traverse(current_dir, root)


def scan_object(p: Path, object_cat: str, object_nr: str, root: str):
    global table_data
    global mymaps_csv
    global newest_plaza_txt_mtime
    global ftp_client

    plaza_txt_mtime: float = 0

    log.debug(f"scan_object {object_cat}{object_nr}")

    path_plaza_txt = p.joinpath("plaza.txt")

    if not path_plaza_txt.exists():
        return

    ### read config file and run migrations of txt-file
    config_object = migrate(path_plaza_txt)

    # obj = {s: dict(config_object.items(s)) for s in config_object.sections()}

    ### check if txt-file is newer
    plaza_txt_mtime = path_plaza_txt.stat().st_mtime
    if plaza_txt_mtime > newest_plaza_txt_mtime:
        newest_plaza_txt_mtime = plaza_txt_mtime

    create_directories(p)

    count_imports = len(list(p.joinpath("Import").glob("*")))
    count_images = len(list(p.joinpath("Publico/Imagen").glob("*")))
    count_videos = len(list(p.joinpath("Publico/Video").glob("*")))
    count_floor_plans = len(list(p.joinpath("Publico/Plano de planta").glob("*")))
    count_slideshows = len(list(p.joinpath("Publico/Slideshow").glob("*")))

    if p.parent.name == "0Archiv":
        status = "archiviert"
        area = p.relative_to(root).parents[1].as_posix().split("/")
    elif p.parent.name == "0Verkauft":
        status = "verkauft"
        area = p.relative_to(root).parents[1].as_posix().split("/")
    elif p.parent.name == "0Pausiert":
        status = "pausiert"
        area = p.relative_to(root).parents[1].as_posix().split("/")
    else:
        status = "aktiv"
        area = p.relative_to(root).parents[0].as_posix().split("/")
        if config_object.get("MAIN", "plaza", fallback="") == "":
            status = "anzulegen"

    if len(area) > 1:
        area = area[1]
    else:
        area = area[0]

    make_exposee(
        p=p,
        plaza_txt_mtime=plaza_txt_mtime,
        id=object_cat + object_nr,
        area=area,
        title=config_object.get("MAIN", "title", fallback=""),
        short_url=config_object.get("MAIN", "short_url", fallback=""),
        price=config_object.get("MAIN", "price", fallback=""),
        living_space=config_object.get("EXPOSEE", "living_space", fallback=""),
        plot_area=config_object.get("EXPOSEE", "plot_area", fallback=""),
        rooms=config_object.get("EXPOSEE", "rooms", fallback=""),
        bedrooms=config_object.get("EXPOSEE", "bedrooms", fallback=""),
        bathrooms=config_object.get("EXPOSEE", "bathrooms", fallback=""),
    )

    make_ad(
        p=p,
        plaza_txt_mtime=plaza_txt_mtime,
        id=object_cat + object_nr,
        title=config_object.get("MAIN", "title", fallback=""),
        short_url=config_object.get("MAIN", "short_url", fallback=""),
        price=config_object.get("MAIN", "price", fallback=""),
    )

    hauptansicht = make_hauptansicht(p)
    if hauptansicht is not None:
        ftp_upload(
            remote_filename=object_cat + object_nr + hauptansicht.suffix,
            local_file_path=hauptansicht,
        )
        hauptansicht = hauptansicht.relative_to(root).as_posix()

    ### position need some conversion
    position = config_object.get("MAIN", "position", fallback="")
    if position != "":
        lon, lat = position.split(",")
        title = config_object.get("MAIN", "title", fallback="")
        mymaps_csv += f'"POINT ({lat} {lon})", {object_cat + object_nr}, {title}\n'

    table_data.append(
        {
            "id": object_cat + object_nr,
            "main_category": object_cat,
            "area": area,
            "directory_path": p.relative_to(root).as_posix(),
            "count_imports": count_imports,
            "count_images": count_images,
            "count_videos": count_videos,
            "count_floor_plans": count_floor_plans,
            "count_slideshows": count_slideshows,
            "status": status,
            "hauptansicht": hauptansicht,
            "title": config_object.get("MAIN", "title", fallback=""),
            "remark": config_object.get("MAIN", "remark", fallback=""),
            "category": config_object.get("MAIN", "category", fallback=""),
            "price": config_object.get("MAIN", "price", fallback=""),
            "rent": config_object.get("MAIN", "rent", fallback=""),
            "position": config_object.get("MAIN", "position", fallback=""),
            "hide_position": config_object.get("MAIN", "hide_position", fallback=""),
            "grafcan": config_object.get("MAIN", "grafcan", fallback=""),
            "plaza": config_object.get("MAIN", "plaza", fallback=""),
            "short_url": config_object.get("MAIN", "short_url", fallback=""),
            "exposee": {
                "altitude": config_object.get("EXPOSEE", "altitude", fallback=""),
                "living_space": config_object.get(
                    "EXPOSEE", "living_space", fallback=""
                ),
                "plot_area": config_object.get("EXPOSEE", "plot_area", fallback=""),
                "rooms": config_object.get("EXPOSEE", "rooms", fallback=""),
                "bedrooms": config_object.get("EXPOSEE", "bedrooms", fallback=""),
                "bathrooms": config_object.get("EXPOSEE", "bathrooms", fallback=""),
            },
            "media": {
                "idealista": config_object.get("MEDIA", "idealista", fallback=""),
                "immoscout": config_object.get("MEDIA", "immoscout", fallback=""),
                "instagram": config_object.get("MEDIA", "instagram", fallback=""),
                "foto_lifting": config_object.get("MEDIA", "foto_lifting", fallback=""),
                "remark": config_object.get("MEDIA", "remark", fallback=""),
            },
            "contract": {
                "valid_until": config_object.get(
                    "CONTRACT", "valid_until", fallback=""
                ),
                "door_key": config_object.get("CONTRACT", "door_key", fallback=""),
                "inhabited": config_object.get("CONTRACT", "inhabited", fallback=""),
            },
            "owner1": {
                "first_name": config_object.get("OWNER1", "first_name", fallback=""),
                "last_name": config_object.get("OWNER1", "last_name", fallback=""),
                "phone": config_object.get("OWNER1", "phone", fallback=""),
                "email": config_object.get("OWNER1", "email", fallback=""),
            },
            "owner2": {
                "first_name": config_object.get("OWNER2", "first_name", fallback=""),
                "last_name": config_object.get("OWNER2", "last_name", fallback=""),
                "phone": config_object.get("OWNER2", "phone", fallback=""),
                "email": config_object.get("OWNER2", "email", fallback=""),
            },
            "housekeeper": {
                "first_name": config_object.get(
                    "HOUSEKEEPER", "first_name", fallback=""
                ),
                "last_name": config_object.get("HOUSEKEEPER", "last_name", fallback=""),
                "phone": config_object.get("HOUSEKEEPER", "phone", fallback=""),
                "email": config_object.get("HOUSEKEEPER", "email", fallback=""),
            },
            "tenant": {
                "first_name": config_object.get("TENANT", "first_name", fallback=""),
                "last_name": config_object.get("TENANT", "last_name", fallback=""),
                "phone": config_object.get("TENANT", "phone", fallback=""),
                "email": config_object.get("TENANT", "email", fallback=""),
            },
        }
    )


def migrate(path_plaza_txt: Path) -> configparser.RawConfigParser:
    PLAZA_VERSION: str = "11"

    config_object = configparser.RawConfigParser()

    with path_plaza_txt.open() as f:
        config_object.read_file(f)

    mtime = path_plaza_txt.stat().st_mtime
    atime = path_plaza_txt.stat().st_atime

    if int(config_object.get("MAIN", "version", fallback="-1")) >= int(PLAZA_VERSION):
        return config_object

    new_config = configparser.RawConfigParser()

    ### bill_ibi, bill_electricity, bill_water, energy_cert, escritura, nota_simple

    new_config.add_section("MAIN")
    new_config["MAIN"]["version"] = PLAZA_VERSION
    new_config["MAIN"]["title"] = config_object.get("MAIN", "title", fallback="")
    new_config["MAIN"]["remark"] = config_object.get("MAIN", "remark", fallback="")
    new_config["MAIN"]["category"] = config_object.get("MAIN", "category", fallback="")
    new_config["MAIN"]["price"] = config_object.get("MAIN", "price", fallback="")
    new_config["MAIN"]["rent"] = config_object.get("MAIN", "rent", fallback="")
    new_config["MAIN"]["position"] = config_object.get("MAIN", "position", fallback="")
    new_config["MAIN"]["hide_position"] = config_object.get(
        "MAIN", "hide_position", fallback="y"
    )
    new_config["MAIN"]["address"] = config_object.get("MAIN", "address", fallback="")
    new_config["MAIN"]["ref_catastral"] = config_object.get(
        "MAIN", "ref_catastral", fallback=""
    )
    new_config["MAIN"]["grafcan"] = config_object.get("MAIN", "grafcan", fallback="")
    new_config["MAIN"]["plaza"] = config_object.get("MAIN", "plaza", fallback="")
    new_config["MAIN"]["short_url"] = config_object.get(
        "MAIN", "short_url", fallback=""
    )
    new_config.add_section("EXPOSEE")
    new_config["EXPOSEE"]["altitude"] = config_object.get(
        "EXPOSEE", "altitude", fallback=""
    )
    new_config["EXPOSEE"]["living_space"] = config_object.get(
        "EXPOSEE", "living_space", fallback=""
    )
    new_config["EXPOSEE"]["plot_area"] = config_object.get(
        "EXPOSEE", "plot_area", fallback=""
    )
    new_config["EXPOSEE"]["rooms"] = config_object.get("EXPOSEE", "rooms", fallback="")
    new_config["EXPOSEE"]["bedrooms"] = config_object.get(
        "EXPOSEE", "bedrooms", fallback=""
    )
    new_config["EXPOSEE"]["bathrooms"] = config_object.get(
        "EXPOSEE", "bathrooms", fallback=""
    )

    new_config.add_section("MEDIA")
    new_config["MEDIA"]["idealista"] = config_object.get(
        "MEDIA", "idealista", fallback=""
    )
    new_config["MEDIA"]["immoscout"] = config_object.get(
        "MEDIA", "immoscout", fallback=""
    )
    new_config["MEDIA"]["instagram"] = config_object.get(
        "MEDIA", "instagram", fallback=""
    )
    new_config["MEDIA"]["foto_lifting"] = config_object.get(
        "MEDIA", "foto_lifting", fallback=""
    )
    new_config["MEDIA"]["remark"] = config_object.get("MEDIA", "remark", fallback="")

    new_config.add_section("CONTRACT")
    new_config["CONTRACT"]["valid_until"] = config_object.get(
        "CONTRACT", "valid_until", fallback=""
    )
    new_config["CONTRACT"]["door_key"] = config_object.get(
        "CONTRACT", "door_key", fallback=""
    )
    new_config["CONTRACT"]["inhabited"] = config_object.get(
        "CONTRACT", "inhabited", fallback=""
    )

    new_config.add_section("HOUSEKEEPER")
    new_config["HOUSEKEEPER"]["first_name"] = config_object.get(
        "HOUSEKEEPER", "first_name", fallback=""
    )
    new_config["HOUSEKEEPER"]["last_name"] = config_object.get(
        "HOUSEKEEPER", "last_name", fallback=""
    )
    new_config["HOUSEKEEPER"]["phone"] = config_object.get(
        "HOUSEKEEPER", "phone", fallback=""
    )
    new_config["HOUSEKEEPER"]["email"] = config_object.get(
        "HOUSEKEEPER", "email", fallback=""
    )

    new_config.add_section("TENANT")
    new_config["TENANT"]["sex"] = config_object.get("TENANT", "sex", fallback="")
    new_config["TENANT"]["first_name"] = config_object.get(
        "TENANT", "first_name", fallback=""
    )
    new_config["TENANT"]["last_name"] = config_object.get(
        "TENANT", "last_name", fallback=""
    )
    new_config["TENANT"]["nationality"] = config_object.get(
        "TENANT", "nationality", fallback=""
    )
    new_config["TENANT"]["town"] = config_object.get("TENANT", "town", fallback="")
    new_config["TENANT"]["zip"] = config_object.get("TENANT", "zip", fallback="")
    new_config["TENANT"]["street"] = config_object.get("TENANT", "street", fallback="")
    new_config["TENANT"]["number"] = config_object.get("TENANT", "number", fallback="")
    new_config["TENANT"]["nation"] = config_object.get("TENANT", "nation", fallback="")
    new_config["TENANT"]["phone"] = config_object.get("TENANT", "phone", fallback="")
    new_config["TENANT"]["email"] = config_object.get("TENANT", "email", fallback="")

    new_config.add_section("OWNER1")
    new_config["OWNER1"]["sex"] = config_object.get("OWNER1", "sex", fallback="")
    new_config["OWNER1"]["first_name"] = config_object.get(
        "OWNER1", "first_name", fallback=""
    )
    new_config["OWNER1"]["last_name"] = config_object.get(
        "OWNER1", "last_name", fallback=""
    )
    new_config["OWNER1"]["nationality"] = config_object.get(
        "OWNER1", "nationality", fallback=""
    )
    new_config["OWNER1"]["town"] = config_object.get("OWNER1", "town", fallback="")
    new_config["OWNER1"]["zip"] = config_object.get("OWNER1", "zip", fallback="")
    new_config["OWNER1"]["street"] = config_object.get("OWNER1", "street", fallback="")
    new_config["OWNER1"]["number"] = config_object.get("OWNER1", "number", fallback="")
    new_config["OWNER1"]["nation"] = config_object.get("OWNER1", "nation", fallback="")
    new_config["OWNER1"]["nie"] = config_object.get("OWNER1", "nie", fallback="")
    new_config["OWNER1"]["idnr"] = config_object.get("OWNER1", "idnr", fallback="")
    new_config["OWNER1"]["phone"] = config_object.get("OWNER1", "phone", fallback="")
    new_config["OWNER1"]["email"] = config_object.get("OWNER1", "email", fallback="")

    new_config.add_section("OWNER2")
    new_config["OWNER2"]["sex"] = config_object.get("OWNER2", "sex", fallback="")
    new_config["OWNER2"]["first_name"] = config_object.get(
        "OWNER2", "first_name", fallback=""
    )
    new_config["OWNER2"]["last_name"] = config_object.get(
        "OWNER2", "last_name", fallback=""
    )
    new_config["OWNER2"]["nationality"] = config_object.get(
        "OWNER2", "nationality", fallback=""
    )
    new_config["OWNER2"]["town"] = config_object.get("OWNER2", "town", fallback="")
    new_config["OWNER2"]["zip"] = config_object.get("OWNER2", "zip", fallback="")
    new_config["OWNER2"]["street"] = config_object.get("OWNER2", "street", fallback="")
    new_config["OWNER2"]["number"] = config_object.get("OWNER2", "number", fallback="")
    new_config["OWNER2"]["nation"] = config_object.get("OWNER2", "nation", fallback="")
    new_config["OWNER2"]["nie"] = config_object.get("OWNER2", "nie", fallback="")
    new_config["OWNER2"]["idnr"] = config_object.get("OWNER2", "idnr", fallback="")
    new_config["OWNER2"]["phone"] = config_object.get("OWNER2", "phone", fallback="")
    new_config["OWNER2"]["email"] = config_object.get("OWNER2", "email", fallback="")

    new_config.add_section("BUYER1")
    new_config["BUYER1"]["sex"] = config_object.get("BUYER1", "sex", fallback="")
    new_config["BUYER1"]["first_name"] = config_object.get(
        "BUYER1", "first_name", fallback=""
    )
    new_config["BUYER1"]["last_name"] = config_object.get(
        "BUYER1", "last_name", fallback=""
    )
    new_config["BUYER1"]["nationality"] = config_object.get(
        "BUYER1", "nationality", fallback=""
    )
    new_config["BUYER1"]["town"] = config_object.get("BUYER1", "town", fallback="")
    new_config["BUYER1"]["zip"] = config_object.get("BUYER1", "zip", fallback="")
    new_config["BUYER1"]["street"] = config_object.get("BUYER1", "street", fallback="")
    new_config["BUYER1"]["number"] = config_object.get("BUYER1", "number", fallback="")
    new_config["BUYER1"]["nation"] = config_object.get("BUYER1", "nation", fallback="")
    new_config["BUYER1"]["nie"] = config_object.get("BUYER1", "nie", fallback="")
    new_config["BUYER1"]["idnr"] = config_object.get("BUYER1", "idnr", fallback="")
    new_config["BUYER1"]["phone"] = config_object.get("BUYER1", "phone", fallback="")
    new_config["BUYER1"]["email"] = config_object.get("BUYER1", "email", fallback="")

    new_config.add_section("BUYER2")
    new_config["BUYER2"]["sex"] = config_object.get("BUYER2", "sex", fallback="")
    new_config["BUYER2"]["first_name"] = config_object.get(
        "BUYER2", "first_name", fallback=""
    )
    new_config["BUYER2"]["last_name"] = config_object.get(
        "BUYER2", "last_name", fallback=""
    )
    new_config["BUYER2"]["nationality"] = config_object.get(
        "BUYER2", "nationality", fallback=""
    )

    new_config["BUYER2"]["town"] = config_object.get("BUYER2", "town", fallback="")
    new_config["BUYER2"]["zip"] = config_object.get("BUYER2", "zip", fallback="")
    new_config["BUYER2"]["street"] = config_object.get("BUYER2", "street", fallback="")
    new_config["BUYER2"]["number"] = config_object.get("BUYER2", "number", fallback="")
    new_config["BUYER2"]["nation"] = config_object.get("BUYER2", "nation", fallback="")
    new_config["BUYER2"]["nie"] = config_object.get("BUYER2", "nie", fallback="")
    new_config["BUYER2"]["idnr"] = config_object.get("BUYER2", "idnr", fallback="")
    new_config["BUYER2"]["phone"] = config_object.get("BUYER2", "phone", fallback="")
    new_config["BUYER2"]["email"] = config_object.get("BUYER2", "email", fallback="")

    new_config.add_section("OPTION")
    new_config["OPTION"]["amount_notary"] = config_object.get(
        "OPTION", "amount_notary", fallback=""
    )
    new_config["OPTION"]["amount_meuble"] = config_object.get(
        "OPTION", "amount_meuble", fallback=""
    )
    new_config["OPTION"]["amount_option"] = config_object.get(
        "OPTION", "amount_option", fallback=""
    )
    new_config["OPTION"]["deadline"] = config_object.get(
        "OPTION", "deadline", fallback=""
    )
    new_config["OPTION"]["deadline"] = config_object.get(
        "OPTION", "deadline", fallback=""
    )
    new_config["OPTION"]["iban"] = config_object.get("OPTION", "iban", fallback="")

    try:
        with path_plaza_txt.open("w") as f:
            new_config.write(f)
    except Exception as e:
        log.exception(e, exc_info=True)

    # Todo: set mtime and atime
    os.utime(path_plaza_txt.as_posix(), times=(atime, mtime))
    return new_config


def create_directories(p: Path):
    #    if p.parent.name not in ("0Archiv", "0Verkauft"):
    #        p.parent.joinpath("0Archiv").mkdir(exist_ok=True)
    #        p.parent.joinpath("0Verkauft").mkdir(exist_ok=True)

    #    if p.joinpath("Publico/Catastro").exists():
    #        try:
    #            p.joinpath("Publico/Catastro").rename(
    #                p.joinpath("Documentos Plaza/Grafcan")
    #            )
    #        except Exception as e:
    #            log.exception(e, exc_info=True)
    #    else:
    #        p.joinpath("Documentos Plaza/Grafcan").mkdir(exist_ok=True, parents=True)

    p.joinpath("Documentos C").mkdir(exist_ok=True)
    p.joinpath("Documentos V").mkdir(exist_ok=True)
    p.joinpath("Documentos Plaza").mkdir(exist_ok=True, parents=True)
    p.joinpath("Documentos Plaza/Grafcan").mkdir(exist_ok=True, parents=True)

    p.joinpath("Import").mkdir(exist_ok=True)

    p.joinpath("Publico/Imagen").mkdir(exist_ok=True, parents=True)
    p.joinpath("Publico/Exposee/Imagen").mkdir(exist_ok=True, parents=True)
    p.joinpath("Publico/Plano de planta").mkdir(exist_ok=True, parents=True)
    p.joinpath("Publico/Video").mkdir(exist_ok=True, parents=True)
    p.joinpath("Publico/Slideshow").mkdir(exist_ok=True, parents=True)


def make_hauptansicht(p: Path) -> Union[Path, None]:
    global newest_plaza_txt_mtime

    ideal_width = 200
    ideal_height = 200
    dest_file = p.joinpath("hauptansicht.jpg")
    source_dir = p.joinpath("Publico/Imagen")

    images = get_images(source_dir)

    if images is None or len(images) < 1:
        return None
    image = images[0]

    if dest_file.exists() and dest_file.stat().st_mtime > image.stat().st_mtime:
        return dest_file

    img = scale_img(image, ideal_width=ideal_width, ideal_height=ideal_height)

    if img is None:
        return None

    try:
        img.save(dest_file)
    except OSError or ValueError:
        return None

    newest_plaza_txt_mtime = True

    return dest_file


def setup_logging(logfile: Union[str, None], debug: Union[int, None] = None):
    handlers: List[logging.Handler] = []
    if debug is not None:
        level = logging.DEBUG
    else:
        level = logging.INFO

    console_handler = logging.StreamHandler()
    console_handler.setFormatter(
        logging.Formatter("%(levelname)s;%(name)s.%(lineno)d;%(message)s")
    )
    handlers.append(console_handler)

    if logfile is not None:
        Path(logfile).parent.mkdir(parents=True, exist_ok=True)
        file_handler = TimedRotatingFileHandler(
            logfile, when="midnight", backupCount=10
        )
        file_handler.setFormatter(
            logging.Formatter(
                "%(asctime)s;%(levelname)s;%(name)s.%(lineno)d;%(message)s"
            )
        )
        handlers.append(file_handler)

    logging.basicConfig(level=level, handlers=handlers)
    logging.getLogger("fontTools.subset").level = logging.WARN
    logging.getLogger("fontTools.ttLib.ttFont").level = logging.WARN
    logging.getLogger("PIL.TiffImagePlugin").level = logging.WARN
    logging.getLogger("fpdf.output").level = logging.WARN


def test():
    pass


if __name__ == "__main__":
    test()
