from pathlib import Path
from importlib import resources
from typing import Union, List
import logging
from fpdf import FPDF
from fpdf.fonts import FontFace
from fpdf.enums import TableCellFillMode, Align


from qrcode import QRCode
from qrcode.constants import ERROR_CORRECT_M

from table_maker.helper import scale_img, get_images, to_cur

log = logging.getLogger(__name__)


def make_exposee(
    p: Path,
    plaza_txt_mtime: float,
    id: str,
    area: str,
    title: str,
    short_url: str,
    price: str,
    living_space: str,
    plot_area: str,
    rooms: str,
    bedrooms: str,
    bathrooms: str,
):
    small_images = get_images(p.joinpath("Publico/Exposee/Imagen"))
    if small_images is None or len(small_images) < 3:
        return None

    if len(small_images) == 4:
        main_image = small_images[0]
        small_images.pop(0)
    else:
        main_image = get_images(p.joinpath("Publico/Imagen"))
        if main_image is None or len(main_image) < 1:
            return None
        main_image = main_image[0]

    price = to_cur(price) + "* EUR"

    try:
        living_space = f"ca. {int(living_space):,}".replace(",", ".") + " m²"
    except ValueError:
        living_space = ""

    try:
        plot_area = f"ca. {int(plot_area):,}".replace(",", ".") + " m²"
    except ValueError:
        plot_area = ""

    for format in ("A4", "A3"):
        path_pdf_exposee = p.joinpath(f"Publico/Exposee/{id}-Exposee-{format}.pdf")
        if (
            not path_pdf_exposee.exists()
            or path_pdf_exposee.stat().st_mtime <= plaza_txt_mtime
        ):
            pdf = get_scaled_pdf(
                id=id,
                title=title,
                short_url=short_url,
                price=price,
                area=area,
                living_space=living_space,
                plot_area=plot_area,
                rooms=rooms,
                bedrooms=bedrooms,
                bathrooms=bathrooms,
                small_images=small_images,
                main_image=main_image,
                format=format,
            )
            if pdf is not None:
                try:
                    pdf.output(path_pdf_exposee.as_posix())
                except Exception as e:
                    log.exception(e, exc_info=True)


def get_scaled_pdf(
    id: str,
    title: str,
    area: str,
    short_url: str,
    price: str,
    living_space: str,
    plot_area: str,
    rooms: str,
    bedrooms: str,
    bathrooms: str,
    small_images: List[Path],
    main_image: Path,
    format: str,
) -> Union[FPDF, None]:
    PPM: int = 5
    a3a4_factor: float = 1

    if format == "A3":
        a3a4_factor = 1.4

    pdf = FPDF(orientation="P", unit="mm", format=format)  # type: ignore
    # Different styles of the same font family.
    with resources.as_file(
        resources.files("table_maker").joinpath("assets/FreeSans.ttf")
    ) as fp:
        pdf.add_font(
            "free-sans",
            style="",
            fname=fp,
        )
    with resources.as_file(
        resources.files("table_maker").joinpath("assets/FreeSansBold.ttf")
    ) as fp:
        pdf.add_font(
            "free-sans",
            style="B",
            fname=fp,
        )
    with resources.as_file(
        resources.files("table_maker").joinpath("assets/FreeSansOblique.ttf")
    ) as fp:
        pdf.add_font(
            "free-sans",
            style="I",
            fname=fp,
        )
    with resources.as_file(
        resources.files("table_maker").joinpath("assets/FreeSansBoldOblique.ttf")
    ) as fp:
        pdf.add_font(
            "free-sans",
            style="BI",
            fname=fp,
        )

    pdf.add_page()
    pdf.set_margin(6)
    pdf.set_margins(left=20 * a3a4_factor, top=20 * a3a4_factor, right=20 * a3a4_factor)

    # Rendering logo:
    with resources.as_file(resources.files("table_maker") / "assets/logo.jpg") as fp:
        pdf.image(
            name=fp.as_posix(),
            x=Align.L,
            y=10 * a3a4_factor,
            # y=10 * a3a4_factor,
            w=60 * a3a4_factor,
        )

    pdf.set_font("free-sans", style="", size=round(24 * a3a4_factor))
    pdf.set_draw_color(119, 2, 21)
    pdf.set_line_width(0.6)
    text = "Venta/Verkauf/Sale"
    # Moving cursor to the right:
    pdf.cell(pdf.epw - pdf.get_string_width(text) - 3)
    pdf.cell(None, None, text, border="B", align=Align.C)

    pdf.ln(20 * a3a4_factor)

    pdf.set_font("free-sans", style="B", size=round(16 * a3a4_factor))
    pdf.set_text_color(255, 255, 255)
    pdf.set_fill_color(119, 2, 21)
    pdf.multi_cell(  # type: ignore
        w=0,
        h=17 * a3a4_factor,
        border=0,
        text=f"{id} {title}",
        new_x="LMARGIN",
        new_y="NEXT",
        align="C",
        fill=True,
        max_line_height=8 * a3a4_factor,
    )
    pdf.ln(2 * a3a4_factor)

    width = int(pdf.epw)
    height = int(width * 0.6)
    img = scale_img(
        main_image,
        width * PPM,
        height * PPM,
    )
    if img is None:
        return None

    pdf.image(img, x=Align.C, y=pdf.y, w=width, h=height)

    pdf.ln(height)

    qr = QRCode(
        version=1,
        error_correction=ERROR_CORRECT_M,
        box_size=15 * a3a4_factor,
        border=4,
    )
    qr.add_data(short_url)
    qr.make(fit=True)
    img_qr = qr.make_image(fill_color="black", back_color="white").get_image()

    pdf.image(img_qr, x=Align.L, y=pdf.y - qr.box_size, w=qr.box_size, h=qr.box_size)

    pdf.set_font("free-sans", style="B", size=round(16 * a3a4_factor))
    pdf.cell(w=(pdf.epw - pdf.get_string_width(price) - 6))
    pdf.set_text_color(0)
    pdf.set_fill_color(255, 153, 102)
    pdf.cell(
        w=pdf.get_string_width(price) + 6,
        h=-10 * a3a4_factor,
        text=price,
        fill=True,
        align="C",
        #        new_x="LMARGIN",
    )

    with pdf.rotation(90, x=pdf.x, y=pdf.y - 2 * a3a4_factor):
        pdf.set_text_color(0)
        pdf.set_font("free-sans", size=round(9 * a3a4_factor))
        pdf.cell(None, None, "* zzgl. Käuferprovision 3% + IGIC", align=Align.R)

    pdf.ln(2 * a3a4_factor)

    width = round(pdf.epw / 3 - 2)
    height = round(width * 0.8)
    img = scale_img(
        small_images[0],
        width * PPM,
        height * PPM,
    )
    if img is None:
        return None

    pdf.image(img, x=Align.L, y=pdf.y, w=width, h=height)

    img = scale_img(
        small_images[1],
        width * PPM,
        height * PPM,
    )
    if img is None:
        return None

    pdf.image(img, x=Align.C, y=pdf.y, w=width, h=height)

    img = scale_img(
        small_images[2],
        width * PPM,
        height * PPM,
    )
    if img is None:
        return None

    pdf.image(img, x=Align.R, y=pdf.y, w=width, h=height)
    pdf.ln(height)

    pdf.ln(2 * a3a4_factor)

    pdf.set_font("free-sans", style="B", size=round(11 * a3a4_factor))
    pdf.set_line_width(0.1 * a3a4_factor)
    pdf.set_draw_color(100)
    pdf.set_text_color(0, 0, 0)
    pdf.set_fill_color(255, 255, 255)
    with pdf.table(
        text_align=("LEFT", "LEFT", "LEFT", "RIGHT"),
        headings_style=FontFace(
            emphasis="BOLD",
            size_pt=round(14 * a3a4_factor),
            color=255,
            fill_color=((119, 2, 21)),
        ),
        #        borders_layout=TableBordersLayout.MINIMAL,
        cell_fill_color=(210),
        cell_fill_mode=TableCellFillMode.ROWS,
        padding=0,
    ) as table:
        row = table.row()
        row.cell("Informacíon de la Propiedad / Daten / Facts", colspan=4, align="c")

        row = table.row()
        row.cell("Municipio")
        row.cell("Ort")
        row.cell("Municipality")
        row.cell(area)

        if living_space != "":
            row = table.row()
            row.cell("Superfice habitable")
            row.cell("Wohnfläche")
            row.cell("Living space")
            row.cell(living_space)

        if plot_area != "":
            row = table.row()
            row.cell("Terreno")
            row.cell("Grundstück")
            row.cell("Plot area")
            row.cell(plot_area)

        if rooms != "":
            row = table.row()
            row.cell("Habitaciones total")
            row.cell("Zimmer gesamt")
            row.cell("Rooms")
            row.cell(rooms)

        if bedrooms != "":
            row = table.row()
            row.cell("Dormitorios")
            row.cell("Schlafzimmer")
            row.cell("Bedrooms")
            row.cell(bedrooms)

        if bathrooms != "":
            row = table.row()
            row.cell("Baños")
            row.cell("Bäder")
            row.cell("Bathrooms")
            row.cell(bathrooms)

    pdf.set_y(-25 * a3a4_factor)
    pdf.set_font("free-sans", size=round(11 * a3a4_factor))
    pdf.set_text_color(119, 2, 21)
    pdf.set_draw_color(119, 2, 21)
    pdf.set_line_width(1 * a3a4_factor)

    pdf.cell(
        w=0,
        h=11 * a3a4_factor,
        text="www.plaza-inmobiliaria.de · +34 922 48 08 16 / +34 664 84 17 89 · anfrage@plaza-inmobiliaria.de",
        border="T",
        align=Align.C,
    )
    return pdf


if __name__ == "__main__":
    make_exposee(
        Path("/home/stephan/Plaza/Objekte/Norden/Garafia/ST680, 440.000 Horst"),
        plaza_txt_mtime=0,
        id="ST680",
        title="Historisches Haus mit Dachterasse und Meerblick\nSanta Domingo de Garafia",
        short_url="https://www.plaza-inmobiliaria.de/immobilie/st680-herrschaftliches-kanarisches-haus-mit-dachterrasse-garafia/",
        price="440000",
        area="Garafia",
        living_space="99",
        plot_area="21170",
        rooms="4",
        bedrooms="2",
        bathrooms="1",
    )
