import logging
from ftplib import FTP_TLS, all_errors
from typing import Union
from pathlib import Path
from datetime import datetime, timezone


log = logging.getLogger(__name__)
ftp_client: Union[FTP_TLS, None] = None


def ftp_setup(host: str, user: str, password: str) -> Union[FTP_TLS, None]:
    global ftp_client

    try:
        ftp_client = FTP_TLS(host=host, user=user, passwd=password, timeout=10)
        ftp_client.prot_p()
    except Exception as e:
        log.error(e)
        ftp_client = None
    else:
        log.debug(f"Ftp connected: {ftp_client.getwelcome()}")

    return ftp_client


def ftp_upload(remote_filename: str, local_file_path: Path) -> Union[FTP_TLS, None]:
    global ftp_client

    if ftp_client is None:
        return None

    if not local_file_path.exists():
        return ftp_client

    try:
        local_timestamp = local_file_path.stat().st_mtime
    except Exception as e:
        log.error(e)
        return ftp_client

    local_timestamp = datetime.fromtimestamp(
        local_timestamp, datetime.now().astimezone().tzinfo
    )

    try:
        remote_timestamp = ftp_client.voidcmd(f"MDTM {remote_filename}")[4:].strip()
    except all_errors:
        remote_timestamp = "19700101000000"

    remote_timestamp = datetime.strptime(remote_timestamp, "%Y%m%d%H%M%S").replace(
        tzinfo=timezone.utc
    )

    if remote_timestamp > local_timestamp:
        return ftp_client

    try:
        fp = local_file_path.open("rb")
    except Exception as e:
        log.error(e)
        return ftp_client

    try:
        ftp_client.storbinary(f"STOR {remote_filename}", fp)
    except all_errors as e:
        log.error(e)
        ftp_client = None
    else:
        log.debug(f"STOR {remote_filename}")

    fp.close()

    return ftp_client


if __name__ == "__main__":
    ret = ftp_setup(
        host="www174.your-server.de",
        user="plazas_0",
        password="Jh6T3F2HZ5gEgvF5",
    )

    if ret is None:
        exit()

    try:
        remote_timestamp = ret.voidcmd("MDTM test0.jpg")
    except all_errors as e:
        print(type(e))
