from PIL import Image, UnidentifiedImageError
from pathlib import Path
from typing import List, Union
import logging

log = logging.getLogger(__name__)


def get_images(p: Path) -> Union[List[Path], None]:
    images: List[Path] = []
    if not p.exists():
        return None

    for item in p.iterdir():
        if item.is_file() and "." not in item.stem:
            images.append(item)
    if len(images) == 0:
        return None
    return sorted(images)


def scale_img(
    path: Path, ideal_width: int, ideal_height: int
) -> Union[Image.Image, None]:
    try:
        img = Image.open(path.as_posix(), "r")
    except UnidentifiedImageError or FileNotFoundError or ValueError or TypeError as e:
        log.exception(e, exc_info=True)
        return None

    (width, height) = img.size

    if height == 0 or width == 0 or ideal_height == 0 or ideal_height == 0:
        return None

    aspect = width / height

    ideal_aspect = ideal_width / ideal_height

    if aspect > ideal_aspect:
        # Then crop the left and right edges:
        new_width = round(ideal_aspect * height)
        offset = (width - new_width) / 2
        cropping = (offset, 0, width - offset, height)
    else:
        # ... crop the top and bottom:
        new_height = round(width / ideal_aspect)
        offset = (height - new_height) / 2
        cropping = (0, offset, width, height - offset)

    return img.crop(cropping).resize((ideal_width, ideal_height))  # type: ignore


def to_cur(s: Union[str, int]) -> str:
    ret = "-"

    if s == "":
        return ret

    if type(s) is str:
        try:
            s = int(s)
        except ValueError as e:
            log.exception(e, exc_info=True)
        else:
            ret = f"{s:,}".replace(",", ".") + ",-"

    return ret
