import asyncio
import base64
import logging
import secrets

import sys
from logging.handlers import TimedRotatingFileHandler
from pathlib import Path

import aiohttp_debugtoolbar
import aiohttp_jinja2

import jinja2
from aiohttp import web
from aiohttp_session import setup as setup_session
from aiohttp_session.cookie_storage import EncryptedCookieStorage
from cryptography import fernet


from riego import __version__, __app_name__
from riego.options import setup_options

from riego.boxes import setup_boxes
from riego.cloud import setup_cloud
from riego.db import setup_db
from riego.model.parameters import setup_parameters
from riego.mqtt import setup_mqtt
from riego.timer import setup_timer
from riego.valves import setup_valves
from riego.web.error_pages import setup_error_pages
from riego.web.routes import setup_routes
from riego.web.security import current_user_ctx_processor
from riego.web.websockets import setup_websockets


def main() -> None:
    options = setup_options(__app_name__, __version__)

    _setup_logging(options=options)

    if sys.version_info >= (3, 8) and options.WindowsSelectorEventLoopPolicy:
        asyncio.DefaultEventLoopPolicy = asyncio.WindowsSelectorEventLoopPolicy  # noqa: E501

    if sys.platform != "win32":
        import uvloop  # pylint: disable=import-error

        asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

    app = web.Application()

    app["version"] = __version__
    app["options"] = options

    db = setup_db(app, options)
    app["db"] = db

    websockets = setup_websockets(app=app, options=options, db=db)
    parameters = setup_parameters(db=db)
    mqtt = setup_mqtt(app=app, options=options)
    setup_boxes(options=options, db=db, mqtt=mqtt)
    valves = setup_valves(options=options, db=db, mqtt=mqtt, websockets=websockets)
    setup_timer(
        app=app, options=options, db=db, mqtt=mqtt, valves=valves, parameters=parameters
    )
    setup_cloud(app=app, parameters=parameters, options=options)

    fernet_key = fernet.Fernet.generate_key()
    secret_key = base64.urlsafe_b64decode(fernet_key)
    setup_session(app, EncryptedCookieStorage(secret_key))

    loader = jinja2.FileSystemLoader(options.http_server_template_dir)
    aiohttp_jinja2.setup(
        app,
        loader=loader,
        # enable_async=True,
        context_processors=[current_user_ctx_processor],
    )

    setup_routes(app, options=options)

    if not options.verbose:
        setup_error_pages(app)

    if options.enable_aiohttp_debug_toolbar:
        aiohttp_debugtoolbar.setup(app, check_host=False, intercept_redirects=False)

    # Put app as subapp under main_app and create an approbiate redirection
    main_app = web.Application()
    main_app["options"] = options

    if parameters.cloud_identifier is None:
        parameters.cloud_identifier = secrets.token_urlsafe(12)

    async def main_app_handler(request):
        return web.HTTPSeeOther(f"/{parameters.cloud_identifier}/")

    main_app.router.add_get("/", main_app_handler)
    main_app.add_subapp(f"/{parameters.cloud_identifier}/", app)

    async def on_startup(app):
        logging.getLogger(__name__).debug("on_startup")
        if app["options"].enable_asyncio_debug:
            asyncio.get_running_loop().set_debug(True)

    async def on_shutdown(app):
        logging.getLogger(__name__).debug("on_shutdown")

    async def on_cleanup(app):
        logging.getLogger(__name__).debug("on_cleanup")

    main_app.on_startup.append(on_startup)
    main_app.on_shutdown.append(on_shutdown)
    main_app.on_cleanup.append(on_cleanup)

    logging.getLogger(__name__).info(f"Start {parameters.cloud_identifier}")

    web.run_app(
        main_app,
        host=options.http_server_bind_address,
        port=options.http_server_bind_port,
        access_log=_setup_access_log(options=options),
        access_log_format='%{X-Real-IP}i %t "%r" %s %b "%{Referer}i" "%{User-Agent}i"',  # noqa: E501
    )


def _setup_access_log(options=None):
    # TODO logging to stderr is always on
    if options.http_access_log_backup_count == -1:
        return None
    Path(options.http_access_log_file).parent.mkdir(parents=True, exist_ok=True)
    file_handler = TimedRotatingFileHandler(
        options.http_access_log_file,
        when="midnight",
        backupCount=options.http_access_log_backup_count,
    )
    # file_handler.setFormatter(logging.Formatter("%(message)s"))
    file_handler.setLevel(logging.DEBUG)

    access_log = logging.getLogger("http_access_log")
    access_log.setLevel(logging.DEBUG)
    access_log.addHandler(file_handler)
    return access_log


def _setup_logging(options=None):
    formatter = logging.Formatter(
        "%(asctime)s;%(levelname)s;%(name)s.%(lineno)d;%(message)s"
    )
    handlers = []

    if options.verbose:
        level = logging.DEBUG
    else:
        level = logging.INFO

    if options.log_stderr:
        stream_handler = logging.StreamHandler(stream=sys.stdout)
        stream_handler.setFormatter(formatter)
        handlers.append(stream_handler)

    if options.log_backup_count > -1:
        Path(options.log_file).parent.mkdir(parents=True, exist_ok=True)
        file_handler = TimedRotatingFileHandler(
            options.log_file, when="midnight", backupCount=options.log_backup_count
        )
        file_handler.setFormatter(formatter)
        handlers.append(file_handler)

    logging.basicConfig(level=level, handlers=handlers)

    if options.enable_gmqtt_debug:
        logging.getLogger("gmqtt").setLevel(logging.DEBUG)
    else:
        logging.getLogger("gmqtt").setLevel(logging.ERROR)

    if options.enable_ssh_debug:
        logging.getLogger("asyncssh").setLevel(logging.DEBUG)
    else:
        logging.getLogger("asyncssh").setLevel(logging.ERROR)
