import asyncio
import sqlite3
import bcrypt
import datetime
from logging import getLogger
from pathlib import Path
from typing import Callable, Coroutine, Union


from .migrations import migrate


_log = getLogger(__name__)

_db: sqlite3.Connection = None
_subscriptions = []


def get_db():
    global _db
    return _db


def setup_db(app, options):
    global _db

    Path(options.db_filename).parent.mkdir(parents=True, exist_ok=True)

    try:
        _db = sqlite3.connect(
            options.db_filename,
            autocommit=True,
            detect_types=sqlite3.PARSE_DECLTYPES,
        )
    except Exception as e:
        _log.error(f"Unable to connect to database: {e}")
        if _db is not None:
            _db.close()
        exit(1)

    app.on_cleanup.append(_db_cleanup)

    def adapt_datetime_iso(val):
        return val.isoformat(sep=" ")

    sqlite3.register_adapter(datetime.datetime, adapt_datetime_iso)

    def convert_datetime(val):
        return datetime.datetime.fromisoformat(val.decode())

    sqlite3.register_converter("datetime", convert_datetime)

    # Useful at startup to detect some database corruption
    (check,) = _db.execute("PRAGMA integrity_check").fetchone()
    if check != "ok":
        print("Integrity check errors", check)

    _db.execute("VACUUM")

    # WAL mode is good for write performance
    _db.execute("PRAGMA journal_mode = wal").fetchone()

    # Foreign keys are off by default, so turn them on
    _db.execute("PRAGMA foreign_keys = ON")

    migrate(_db)

    _db.row_factory = sqlite3.Row

    if options.reset_admin:
        _reset_admin(options)
        _db.close()
        exit(0)

    _db.create_function("signal_changes", 4, _run_subscriptions)
    return _db


def subscribe_db(callback: Union[Coroutine, Callable], table: str = None) -> None:
    global _subscriptions
    _subscriptions.append({"callback": callback, "table": table})


async def _db_cleanup(app):
    global _db
    if _db is None:
        return
    try:
        _db.close()
    except Exception as e:
        _log.error(f"Unable to close Database: {e}")
    else:
        _log.debug("Database connection closed")


def _run_subscriptions(table: str, action: str, old_rowid: int, new_rowid: int):
    global _subscriptions

    for subscription in _subscriptions:
        if (subscription["table"] != table) & (subscription["table"] is not None):
            continue
        func = subscription["callback"]
        if asyncio.iscoroutinefunction(func):
            loop = asyncio.get_event_loop()
            try:
                loop.create_task(func(table, action, old_rowid, new_rowid))
            except Exception as e:
                _log.error(f"Exeption {e} in awaitable {func}")
        else:
            try:
                func(table, action, old_rowid, new_rowid)
            except Exception as e:
                _log.error(f"Exeption {e} in callable {func}")


def _reset_admin(options):
    password = options.reset_admin

    if len(password) > 0:
        password = password.encode("utf-8")
        password = bcrypt.hashpw(password, bcrypt.gensalt())
        password = password.decode("utf-8")
        cursor = _db.cursor()
        cursor.execute(
            """UPDATE users
                        SET password = ?
                        WHERE id = ? """,
            (password, 1),
        )

        if cursor.rowcount == 1:
            print("Succesfully reset Admin PW: {}".format(password.encode("utf-8")))
        else:
            print("Unable to reset Admin PW:")
