import configargparse
import socket
import pkg_resources
from pathlib import Path


options = None


def get_options():
    global options
    return options


def setup_options(app_name, version):
    global options

    p = configargparse.ArgParser(
        default_config_files=[
            f"/etc/{app_name}/conf.d/*.conf",
            f"~/.{app_name}.conf",
            f"{app_name}.conf",
        ],
        args_for_writing_out_config_file=["-w", "--write-out-config-file"],
    )

    p.add(
        "--config",
        is_config_file=True,
        env_var=app_name.capitalize(),
        required=False,
        help="config file path",
    )

    # Database
    p.add(
        "--db_filename", help="Path and name for DB file", default=f"db/{app_name}.db"
    )
    p.add(
        "--db_migrations_dir",
        help="path to database migrations directory",
        default=pkg_resources.resource_filename(app_name, "migrations"),
    )
    # Logging
    p.add("--log_file", help="Full path to logfile", default=f"log/{app_name}.log")
    p.add(
        "--log_backup_count",
        help="How many files to rotate, -1 to disable",
        default=-1,
        type=int,
    )
    p.add("--log_stderr", help="log to stderr", action="store_true", default=True)
    # Secrets & Scurity
    p.add("--max_age_remember_me", type=int, default=7776000)
    p.add("--cookie_name_remember_me", default="remember_me")
    p.add("--reset_admin", help="Reset admin-pw to given value an exit")
    p.add("--websockets_max_receive_size", type=int, default=1024)
    p.add("--ssh_key_algorithm", default="ssh-ed25519")
    # Session keys
    p.add("--session_key_websocket_auth", default="websocket_auth")
    # TODO
    p.add("--session_key_user_id", default="user_id", help="# TODO not used Yet")
    # HTTP-Server
    p.add(
        "--http_server_bind_address", help="http-server bind address", default="0.0.0.0"
    )
    p.add(
        "--http_server_bind_port", help="http-server bind port", default=8080, type=int
    )
    p.add(
        "--http_server_static_dir",
        help="Serve static html files from this directory",
        default=pkg_resources.resource_filename(f"{app_name}.web", "static"),
    )
    p.add(
        "--http_server_template_dir",
        help="Serve template files from this directory",
        default=pkg_resources.resource_filename(f"{app_name}.web", "templates"),
    )
    p.add("--websocket_path", help="url path for websocket", default="/ws")
    p.add(
        "--http_access_log_file",
        help="Full path to access logfile",
        default="log/access.log",
    )
    p.add(
        "--http_access_log_backup_count",
        help="How many files to rotate, -1 to disable",
        default=-1,
        type=int,
    )
    # Cloud-Server
    #    p.add('--cloud_identifier', help='Unique id for Cloud Identity')
    p.add(
        "--cloud_api_url",
        help="Hostname for cloud service",
        default="https://api.riego.cloud/api_20210221/",
    )
    p.add(
        "--ssh_known_hosts",
        help="SSH-Server verification",
        default=pkg_resources.resource_filename(f"{app_name}", "ssh/known_hosts"),
    )

    # Directories
    p.add(
        "--base_dir",
        help="Change only if you know what you are doing",
        default=Path(__file__).parent,
    )

    # MQTT-Broker
    p.add(
        "--mqtt_broker_host", help="IP adress of MQTT-Broker Host", default="127.0.0.1"
    )
    p.add("--mqtt_broker_port", help="Port of MQTT-Broker Host", default=1883, type=int)
    p.add(
        "--mqtt_client_id",
        help="Client ID for MQTT-Connection",
        default=f"controler_{socket.gethostname()}",
    )
    p.add(
        "--mqtt_keepalive", help="Send keepalive every x seconds", default=60, type=int
    )
    p.add("--mqtt_version", help="MQTTv311 = 4, MQTTv50 = 5", default=4, type=int)
    p.add(
        "--mqtt_result_subscription",
        help="used by class valves",
        default="stat/+/RESULT",
    )
    p.add("--mqtt_lwt_subscription", help="used by class boxes", default="tele/+/LWT")
    p.add(
        "--mqtt_state_subscription", help="used by class boxes", default="tele/+/STATE"
    )
    p.add(
        "--mqtt_info1_subscription", help="used by class boxes", default="tele/+/INFO1"
    )
    p.add(
        "--mqtt_info2_subscription", help="used by class boxes", default="tele/+/INFO2"
    )
    p.add("--mqtt_sensor_subscription", help="yet not used", default="tele/+/SENSOR")
    p.add("--mqtt_cmnd_prefix", help="", default="cmnd")
    p.add("--mqtt_keyword_ON", help="", default="ON")
    p.add("--mqtt_keyword_OFF", help="", default="OFF")
    # Parameter Default Values
    p.add("--parameters_smtp_hostname", default="smtp.gmail.com")
    p.add("--parameters_smtp_port", type=int, default=465)
    p.add("--parameters_smtp_security", default="use_tls=True")
    p.add("--parameters_smtp_user", default="user")
    p.add("--parameters_smtp_password", default="password")
    p.add("--parameters_smtp_from", default="riego@localhost")

    p.add("--parameters_start_time_1", default="19:00")
    p.add("--parameters_max_duartion", default="240")
    # Debug
    p.add("--enable_aiohttp_debug_toolbar", action="store_true")
    p.add("--enable_asyncio_debug", action="store_true")
    p.add("--enable_ssh_debug", action="store_true")
    p.add("--enable_gmqtt_debug", action="store_true")
    p.add("--enable_timer_dev_mode", action="store_true")
    p.add("--WindowsSelectorEventLoopPolicy", action="store_true")

    # Version, Help, Verbosity
    p.add("-v", "--verbose", help="verbose", action="store_true")
    p.add("--version", help="Print version and exit", action="store_true")
    p.add(
        "--defaults",
        help="Print options with default values and exit",
        action="store_true",
    )

    options = p.parse_args()
    if options.verbose:
        print(p.format_values())

    if options.defaults:
        for item in vars(options):
            print(f"# {item}={getattr(options, item)}")
        exit(0)

    if options.version:
        print("Version: ", version)
        exit(0)

    return options
