import asyncio
import secrets
from logging import getLogger
from sqlite3 import IntegrityError

import aiohttp_jinja2
import bcrypt
from aiohttp import web
from aiohttp_session import get_session, new_session

from riego.db import get_db
from riego.web.security import delete_websocket_auth, get_user

_log = getLogger(__name__)


router = web.RouteTableDef()


def setup_routes_login(app):
    app.add_routes(router)


@router.get("/login", name="login")
@aiohttp_jinja2.template("login/login.html")
async def login(request: web.Request):
    redirect = request.rel_url.query.get("redirect", "")
    csrf_token = secrets.token_urlsafe()
    session = await new_session(request)
    # session = await get_session(request)
    session["csrf_token"] = csrf_token
    session.changed()
    return {"csrf_token": csrf_token, "redirect": redirect}


@router.post("/login")
async def login_apply(request: web.Request):
    db = get_db()
    form = await request.post()
    session = await get_session(request)
    if session.get("csrf_token") != form["csrf_token"]:
        # Normally not possible
        await asyncio.sleep(2)
        return web.HTTPUnauthorized()

    if form.get("identity") is None:
        await asyncio.sleep(2)
        return web.HTTPSeeOther(request.app.router["login"].url_for())

    cursor = db.cursor()
    cursor.execute(
        """SELECT *, 'login' AS provider
                    FROM users
                    WHERE identity = ?""",
        (form["identity"],),
    )
    user = cursor.fetchone()

    if (
        user is None
        or user["is_disabled"]
        or user["password"] is None
        or not len(user["password"])
    ):
        await asyncio.sleep(2)
        return web.HTTPSeeOther(request.app.router["login"].url_for())
    if not bcrypt.checkpw(
        form["password"].encode("utf-8"), user["password"].encode("utf-8")
    ):
        await asyncio.sleep(2)
        return web.HTTPSeeOther(request.app.router["login"].url_for())

    session["user_id"] = user["id"]

    location = form.get("redirect")
    if location is None or location == "":
        location = request.app.router["home"].url_for()
    response = web.HTTPSeeOther(location=location)
    # TODO use create_remember_me
    if form.get("remember_me") is not None:
        remember_me = secrets.token_urlsafe()
        try:
            with db:
                db.execute(
                    """UPDATE users
                       SET remember_me = ?
                       WHERE id = ? """,
                    (remember_me, user["id"]),
                )
        except IntegrityError:
            pass
        response.set_cookie(
            "remember_me",
            remember_me,
            max_age=request.app["options"].max_age_remember_me,
            httponly=True,
            samesite="strict",
        )
    return response


@router.get("/logout", name="logout")
async def logout(request: web.Request):
    db = get_db()
    user = await get_user(request)
    if user is not None:
        await delete_websocket_auth(request, user=user)
        # TODO use delete_remember_me
        try:
            with db:
                db.execute(
                    """UPDATE users
                                        SET remember_me = ''
                                        WHERE id = ?""",
                    (user["id"],),
                )
        except IntegrityError:
            pass
    session = await get_session(request)
    if session is not None:
        session.pop("user_id", None)
    response = web.HTTPSeeOther(request.app.router["login"].url_for())
    #    response.set_cookie('remember_me', None,
    #                        expires='Thu, 01 Jan 1970 00:00:00 GMT')
    response.del_cookie("remember_me")
    return response
