"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jenkinsArtifactsBounds = exports.JenkinsProvider = exports.BaseJenkinsProvider = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const cdk = require("@aws-cdk/core");
const custom_action_registration_1 = require("../custom-action-registration");
class BaseJenkinsProvider extends cdk.Construct {
    constructor(scope, id, version) {
        super(scope, id);
        this.version = version || '1';
    }
}
exports.BaseJenkinsProvider = BaseJenkinsProvider;
/**
 * A class representing Jenkins providers.
 *
 * @see #import
 */
class JenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.buildIncluded = false;
        this.testIncluded = false;
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
        if (props.forBuild === true) {
            this._registerBuildProvider();
        }
        if (props.forTest === true) {
            this._registerTestProvider();
        }
    }
    /**
     * Import a Jenkins provider registered either outside the CDK,
     * or in a different CDK Stack.
     *
     * @param scope the parent Construct for the new provider
     * @param id the identifier of the new provider Construct
     * @param attrs the properties used to identify the existing provider
     * @returns a new Construct representing a reference to an existing Jenkins provider
     */
    static fromJenkinsProviderAttributes(scope, id, attrs) {
        return new ImportedJenkinsProvider(scope, id, attrs);
    }
    /**
     * @internal
     */
    _registerBuildProvider() {
        if (this.buildIncluded) {
            return;
        }
        this.buildIncluded = true;
        this.registerJenkinsCustomAction('JenkinsBuildProviderResource', codepipeline.ActionCategory.BUILD);
    }
    /**
     * @internal
     */
    _registerTestProvider() {
        if (this.testIncluded) {
            return;
        }
        this.testIncluded = true;
        this.registerJenkinsCustomAction('JenkinsTestProviderResource', codepipeline.ActionCategory.TEST);
    }
    registerJenkinsCustomAction(id, category) {
        new custom_action_registration_1.CustomActionRegistration(this, id, {
            category,
            artifactBounds: exports.jenkinsArtifactsBounds,
            provider: this.providerName,
            version: this.version,
            entityUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}'),
            executionUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}/{ExternalExecutionId}'),
            actionProperties: [
                {
                    name: 'ProjectName',
                    required: true,
                    key: true,
                    queryable: true,
                },
            ],
        });
    }
}
exports.JenkinsProvider = JenkinsProvider;
class ImportedJenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
    }
    _registerBuildProvider() {
        // do nothing
    }
    _registerTestProvider() {
        // do nothing
    }
}
function appendToUrl(baseUrl, path) {
    return baseUrl.endsWith('/') ? baseUrl + path : `${baseUrl}/${path}`;
}
exports.jenkinsArtifactsBounds = {
    minInputs: 0,
    maxInputs: 5,
    minOutputs: 0,
    maxOutputs: 5,
};
//# sourceMappingURL=data:application/json;base64,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