"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvokeAction = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * CodePipeline invoke Action that is provided by an AWS Lambda function.
 *
 * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-invoke-lambda-function.html
 */
class LambdaInvokeAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.lambda,
            category: codepipeline.ActionCategory.INVOKE,
            provider: 'Lambda',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 5,
                minOutputs: 0,
                maxOutputs: 5,
            },
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the Lambda function this action points to.
     * Variables in Lambda invoke actions are defined by calling the PutJobSuccessResult CodePipeline API call
     * with the 'outputVariables' property filled.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'outputVariables' section of the PutJobSuccessResult
     *   request that the Lambda function calls when the action is invoked
     *
     * @see https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobSuccessResult.html
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        // allow pipeline to list functions
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:ListFunctions'],
            resources: ['*'],
        }));
        // allow pipeline to invoke this lambda functionn
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [this.props.lambda.functionArn],
        }));
        // allow the Role access to the Bucket, if there are any inputs/outputs
        if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(options.role);
        }
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantWrite(options.role);
        }
        // allow lambda to put job results for this pipeline
        // CodePipeline requires this to be granted to '*'
        // (the Pipeline ARN will not be enough)
        this.props.lambda.addToRolePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['codepipeline:PutJobSuccessResult', 'codepipeline:PutJobFailureResult'],
        }));
        return {
            configuration: {
                FunctionName: this.props.lambda.functionName,
                UserParameters: core_1.Stack.of(scope).toJsonString(this.props.userParameters),
            },
        };
    }
}
exports.LambdaInvokeAction = LambdaInvokeAction;
//# sourceMappingURL=data:application/json;base64,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