"use strict";
const assert_1 = require("@aws-cdk/assert");
const codedeploy = require("@aws-cdk/aws-codedeploy");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const cdk = require("@aws-cdk/core");
const cpactions = require("../../lib");
function addEcsDeploymentGroup(stack) {
    return codedeploy.EcsDeploymentGroup.fromEcsDeploymentGroupAttributes(stack, 'EDG', {
        application: codedeploy.EcsApplication.fromEcsApplicationName(stack, 'EA', 'MyApplication'),
        deploymentGroupName: 'MyDeploymentGroup',
    });
}
function addCodeDeployECSCodePipeline(stack, props) {
    new codepipeline.Pipeline(stack, 'Pipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [
                    new cpactions.GitHubSourceAction({
                        actionName: 'GitHub',
                        output: props.taskDefinitionTemplateInput || props.taskDefinitionTemplateFile.artifact,
                        oauthToken: cdk.SecretValue.plainText('secret'),
                        owner: 'awslabs',
                        repo: 'aws-cdk',
                    }),
                    new cpactions.GitHubSourceAction({
                        actionName: 'GitHub2',
                        output: props.appSpecTemplateInput || props.appSpecTemplateFile.artifact,
                        oauthToken: cdk.SecretValue.plainText('secret'),
                        owner: 'awslabs',
                        repo: 'aws-cdk-2',
                    }),
                ],
            },
            {
                stageName: 'Invoke',
                actions: [
                    new cpactions.CodeDeployEcsDeployAction(props),
                ],
            },
        ],
    });
}
module.exports = {
    'CodeDeploy ECS Deploy Action': {
        'throws an exception if more than 4 container image inputs are provided'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            const containerImageInputs = [];
            for (let i = 0; i < 5; i++) {
                containerImageInputs.push({
                    input: artifact,
                });
            }
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                    appSpecTemplateInput: artifact,
                    containerImageInputs,
                });
            }, /Action cannot have more than 4 container image inputs, got: 5/);
            test.done();
        },
        'throws an exception if both appspec artifact input and file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            const artifactPath = new codepipeline.ArtifactPath(artifact, 'hello');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                    appSpecTemplateInput: artifact,
                    appSpecTemplateFile: artifactPath,
                });
            }, /Exactly one of 'appSpecTemplateInput' or 'appSpecTemplateFile' can be provided in the ECS CodeDeploy Action/);
            test.done();
        },
        'throws an exception if neither appspec artifact input nor file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                });
            }, /Specifying one of 'appSpecTemplateInput' or 'appSpecTemplateFile' is required for the ECS CodeDeploy Action/);
            test.done();
        },
        'throws an exception if both task definition artifact input and file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            const artifactPath = new codepipeline.ArtifactPath(artifact, 'hello');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    taskDefinitionTemplateInput: artifact,
                    taskDefinitionTemplateFile: artifactPath,
                    appSpecTemplateInput: artifact,
                });
            }, /Exactly one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' can be provided in the ECS CodeDeploy Action/);
            test.done();
        },
        'throws an exception if neither task definition artifact input nor file are specified'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact = new codepipeline.Artifact('Artifact');
            test.throws(() => {
                new cpactions.CodeDeployEcsDeployAction({
                    actionName: 'DeployToECS',
                    deploymentGroup,
                    appSpecTemplateInput: artifact,
                });
            }, /Specifying one of 'taskDefinitionTemplateInput' or 'taskDefinitionTemplateFile' is required for the ECS CodeDeploy Action/);
            test.done();
        },
        'defaults task definition and appspec template paths'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            addCodeDeployECSCodePipeline(stack, {
                actionName: 'DeployToECS',
                deploymentGroup,
                taskDefinitionTemplateInput: new codepipeline.Artifact('TaskDefArtifact'),
                appSpecTemplateInput: new codepipeline.Artifact('AppSpecArtifact'),
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                Stages: [
                    {},
                    {
                        Actions: [
                            {
                                Configuration: {
                                    ApplicationName: 'MyApplication',
                                    DeploymentGroupName: 'MyDeploymentGroup',
                                    TaskDefinitionTemplateArtifact: 'TaskDefArtifact',
                                    AppSpecTemplateArtifact: 'AppSpecArtifact',
                                    TaskDefinitionTemplatePath: 'taskdef.json',
                                    AppSpecTemplatePath: 'appspec.yaml',
                                },
                                InputArtifacts: [
                                    {
                                        Name: 'TaskDefArtifact',
                                    },
                                    {
                                        Name: 'AppSpecArtifact',
                                    },
                                ],
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'defaults task definition placeholder string'(test) {
            const stack = new cdk.Stack();
            const deploymentGroup = addEcsDeploymentGroup(stack);
            const artifact1 = new codepipeline.Artifact('Artifact1');
            const artifact2 = new codepipeline.Artifact('Artifact2');
            addCodeDeployECSCodePipeline(stack, {
                actionName: 'DeployToECS',
                deploymentGroup,
                taskDefinitionTemplateFile: new codepipeline.ArtifactPath(artifact1, 'task-definition.json'),
                appSpecTemplateFile: new codepipeline.ArtifactPath(artifact2, 'appspec-test.yaml'),
                containerImageInputs: [
                    {
                        input: artifact1,
                    },
                    {
                        input: artifact2,
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                Stages: [
                    {},
                    {
                        Actions: [
                            {
                                Configuration: {
                                    ApplicationName: 'MyApplication',
                                    DeploymentGroupName: 'MyDeploymentGroup',
                                    TaskDefinitionTemplateArtifact: 'Artifact1',
                                    AppSpecTemplateArtifact: 'Artifact2',
                                    TaskDefinitionTemplatePath: 'task-definition.json',
                                    AppSpecTemplatePath: 'appspec-test.yaml',
                                    Image1ArtifactName: 'Artifact1',
                                    Image1ContainerName: 'IMAGE',
                                    Image2ArtifactName: 'Artifact2',
                                    Image2ContainerName: 'IMAGE',
                                },
                                InputArtifacts: [
                                    {
                                        Name: 'Artifact1',
                                    },
                                    {
                                        Name: 'Artifact2',
                                    },
                                ],
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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