"use strict";
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function stackIncludingLambdaInvokeCodePipeline(props) {
    const stack = new core_1.Stack();
    new codepipeline.Pipeline(stack, 'Pipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [
                    new cpactions.GitHubSourceAction({
                        actionName: 'GitHub',
                        output: props.lambdaInput || new codepipeline.Artifact(),
                        oauthToken: core_1.SecretValue.plainText('secret'),
                        owner: 'awslabs',
                        repo: 'aws-cdk',
                    }),
                ],
            },
            {
                stageName: 'Invoke',
                actions: [
                    new cpactions.LambdaInvokeAction({
                        actionName: 'Lambda',
                        lambda: new lambda.Function(stack, 'Lambda', {
                            code: lambda.Code.fromCfnParameters(),
                            handler: 'index.handler',
                            runtime: lambda.Runtime.NODEJS_10_X,
                        }),
                        userParameters: props.userParams,
                        inputs: props.lambdaInput ? [props.lambdaInput] : undefined,
                        outputs: props.lambdaOutput ? [props.lambdaOutput] : undefined,
                    }),
                ],
            },
        ],
    });
    return stack;
}
module.exports = {
    'Lambda invoke Action': {
        'properly serializes the object passed in userParameters'(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                userParams: {
                    key: 1234,
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'UserParameters': '{"key":1234}',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'properly resolves any Tokens passed in userParameters'(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                userParams: {
                    key: core_1.Lazy.stringValue({ produce: () => core_1.Aws.REGION }),
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'UserParameters': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                '{"key":"',
                                                {
                                                    'Ref': 'AWS::Region',
                                                },
                                                '"}',
                                            ],
                                        ],
                                    },
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'properly resolves any stringified Tokens passed in userParameters'(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                userParams: {
                    key: core_1.Token.asString(null),
                },
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'UserParameters': '{"key":null}',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        "assigns the Action's Role with read permissions to the Bucket if it has only inputs"(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                lambdaInput: new codepipeline.Artifact(),
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': 'lambda:ListFunctions',
                            'Resource': '*',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': 'lambda:InvokeFunction',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:DescribeKey',
                            ],
                            'Effect': 'Allow',
                        },
                    ],
                },
            }));
            test.done();
        },
        "assigns the Action's Role with write permissions to the Bucket if it has only outputs"(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                lambdaOutput: new codepipeline.Artifact(),
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': 'lambda:ListFunctions',
                            'Resource': '*',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': 'lambda:InvokeFunction',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                's3:DeleteObject*',
                                's3:PutObject*',
                                's3:Abort*',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                'kms:Encrypt',
                                'kms:ReEncrypt*',
                                'kms:GenerateDataKey*',
                            ],
                            'Effect': 'Allow',
                        },
                    ],
                },
            }));
            test.done();
        },
        "assigns the Action's Role with read-write permissions to the Bucket if it has both inputs and outputs"(test) {
            const stack = stackIncludingLambdaInvokeCodePipeline({
                lambdaInput: new codepipeline.Artifact(),
                lambdaOutput: new codepipeline.Artifact(),
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Action': 'lambda:ListFunctions',
                            'Resource': '*',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': 'lambda:InvokeFunction',
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                'kms:Decrypt',
                                'kms:DescribeKey',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                's3:DeleteObject*',
                                's3:PutObject*',
                                's3:Abort*',
                            ],
                            'Effect': 'Allow',
                        },
                        {
                            'Action': [
                                'kms:Encrypt',
                                'kms:ReEncrypt*',
                                'kms:GenerateDataKey*',
                            ],
                            'Effect': 'Allow',
                        },
                    ],
                },
            }));
            test.done();
        },
        'exposes variables for other actions to consume'(test) {
            const stack = new core_1.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const lambdaInvokeAction = new cpactions.LambdaInvokeAction({
                actionName: 'LambdaInvoke',
                lambda: lambda.Function.fromFunctionArn(stack, 'Func', 'arn:aws:lambda:us-east-1:123456789012:function:some-func'),
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new cpactions.S3SourceAction({
                                actionName: 'S3_Source',
                                bucket: s3.Bucket.fromBucketName(stack, 'Bucket', 'bucket'),
                                bucketKey: 'key',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Invoke',
                        actions: [
                            lambdaInvokeAction,
                            new cpactions.ManualApprovalAction({
                                actionName: 'Approve',
                                additionalInformation: lambdaInvokeAction.variable('SomeVar'),
                                notificationTopic: sns.Topic.fromTopicArn(stack, 'Topic', 'arn:aws:sns:us-east-1:123456789012:mytopic'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Invoke',
                        'Actions': [
                            {
                                'Name': 'LambdaInvoke',
                                'Namespace': 'Invoke_LambdaInvoke_NS',
                            },
                            {
                                'Name': 'Approve',
                                'Configuration': {
                                    'CustomData': '#{Invoke_LambdaInvoke_NS.SomeVar}',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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