"use strict";
const assert_1 = require("@aws-cdk/assert");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function minimalPipeline(stack, options = {}) {
    const sourceOutput = new codepipeline.Artifact();
    const sourceAction = new cpactions.GitHubSourceAction({
        actionName: 'Source',
        owner: 'aws',
        repo: 'aws-cdk',
        output: sourceOutput,
        oauthToken: core_1.SecretValue.plainText('secret'),
    });
    const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline', {
        stages: [
            {
                stageName: 'Source',
                actions: [sourceAction],
            },
        ],
    });
    const deploymentStage = pipeline.addStage({
        stageName: 'Deploy',
        actions: [
            new cpactions.S3DeployAction({
                accessControl: options.accessControl,
                actionName: 'CopyFiles',
                bucket: options.bucket || new s3.Bucket(stack, 'MyBucket'),
                cacheControl: options.cacheControl,
                extract: options.extract,
                input: sourceOutput,
                objectKey: options.objectKey,
            }),
        ],
    });
    return deploymentStage;
}
module.exports = {
    'S3 Deploy Action': {
        'by default extract artifacts'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                        'Actions': [
                            {
                                'Name': 'Source',
                                'ActionTypeId': {
                                    'Category': 'Source',
                                    'Owner': 'ThirdParty',
                                },
                            },
                        ],
                    },
                    {
                        'Name': 'Deploy',
                        'Actions': [
                            {
                                'ActionTypeId': {
                                    'Category': 'Deploy',
                                    'Provider': 'S3',
                                },
                                'Configuration': {
                                    'Extract': 'true',
                                },
                                'Name': 'CopyFiles',
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'grant the pipeline correct access to the target bucket'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                'PolicyDocument': {
                    'Statement': [
                        {
                            'Effect': 'Allow',
                            'Action': [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                                's3:DeleteObject*',
                                's3:PutObject*',
                                's3:Abort*',
                            ],
                        },
                        {},
                        {
                            'Effect': 'Allow',
                            'Action': 'sts:AssumeRole',
                        },
                    ],
                },
            }));
            test.done();
        },
        'kebab-case CannedACL value'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                accessControl: s3.BucketAccessControl.PUBLIC_READ_WRITE,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'CannedACL': 'public-read-write',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'allow customizing cache-control'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                cacheControl: [
                    cpactions.CacheControl.setPublic(),
                    cpactions.CacheControl.maxAge(core_1.Duration.hours(12)),
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'CacheControl': 'public, max-age: 43200',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
        'allow customizing objectKey (deployment path on S3)'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, {
                objectKey: '/a/b/c',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {},
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'ObjectKey': '/a/b/c',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zMy1kZXBsb3ktYWN0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5zMy1kZXBsb3ktYWN0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBMkQ7QUFDM0QsMERBQTBEO0FBQzFELHNDQUFzQztBQUN0Qyx3Q0FBNkQ7QUFFN0QsdUNBQXVDO0FBZ0t2QyxTQUFTLGVBQWUsQ0FBQyxLQUFZLEVBQUUsVUFBa0MsRUFBRTtJQUN6RSxNQUFNLFlBQVksR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztJQUNqRCxNQUFNLFlBQVksR0FBRyxJQUFJLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQztRQUNwRCxVQUFVLEVBQUUsUUFBUTtRQUNwQixLQUFLLEVBQUUsS0FBSztRQUNaLElBQUksRUFBRSxTQUFTO1FBQ2YsTUFBTSxFQUFFLFlBQVk7UUFDcEIsVUFBVSxFQUFFLGtCQUFXLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQztLQUM1QyxDQUFDLENBQUM7SUFFSCxNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRTtRQUM5RCxNQUFNLEVBQUU7WUFDTjtnQkFDRSxTQUFTLEVBQUUsUUFBUTtnQkFDbkIsT0FBTyxFQUFFLENBQUUsWUFBWSxDQUFFO2FBQzFCO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLGVBQWUsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDO1FBQ3hDLFNBQVMsRUFBRSxRQUFRO1FBQ25CLE9BQU8sRUFBRTtZQUNQLElBQUksU0FBUyxDQUFDLGNBQWMsQ0FBQztnQkFDM0IsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhO2dCQUNwQyxVQUFVLEVBQUUsV0FBVztnQkFDdkIsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLElBQUksSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUM7Z0JBQzFELFlBQVksRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDbEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPO2dCQUN4QixLQUFLLEVBQUUsWUFBWTtnQkFDbkIsU0FBUyxFQUFFLE9BQU8sQ0FBQyxTQUFTO2FBQzdCLENBQUM7U0FDSDtLQUNGLENBQUMsQ0FBQztJQUVILE9BQU8sZUFBZSxDQUFDO0FBQ3pCLENBQUM7QUEvTEQsaUJBQVM7SUFDUCxrQkFBa0IsRUFBRTtRQUNsQiw4QkFBOEIsQ0FBQyxJQUFVO1lBQ3ZDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsZUFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRXZCLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsNkJBQTZCLEVBQUU7Z0JBQy9ELFFBQVEsRUFBRTtvQkFDUjt3QkFDRSxNQUFNLEVBQUUsUUFBUTt3QkFDaEIsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLE1BQU0sRUFBRSxRQUFRO2dDQUNoQixjQUFjLEVBQUU7b0NBQ2QsVUFBVSxFQUFFLFFBQVE7b0NBQ3BCLE9BQU8sRUFBRSxZQUFZO2lDQUN0Qjs2QkFDRjt5QkFDRjtxQkFDRjtvQkFDRDt3QkFDRSxNQUFNLEVBQUUsUUFBUTt3QkFDaEIsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLGNBQWMsRUFBRTtvQ0FDZCxVQUFVLEVBQUUsUUFBUTtvQ0FDcEIsVUFBVSxFQUFFLElBQUk7aUNBQ2pCO2dDQUNELGVBQWUsRUFBRTtvQ0FDZixTQUFTLEVBQUUsTUFBTTtpQ0FDbEI7Z0NBQ0QsTUFBTSxFQUFFLFdBQVc7NkJBQ3BCO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsd0RBQXdELENBQUMsSUFBVTtZQUNqRSxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLGVBQWUsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUV2QixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGtCQUFrQixFQUFFO2dCQUNwRCxnQkFBZ0IsRUFBRTtvQkFDaEIsV0FBVyxFQUFFO3dCQUNYOzRCQUNFLFFBQVEsRUFBRSxPQUFPOzRCQUNqQixRQUFRLEVBQUU7Z0NBQ1IsZUFBZTtnQ0FDZixlQUFlO2dDQUNmLFVBQVU7Z0NBQ1Ysa0JBQWtCO2dDQUNsQixlQUFlO2dDQUNmLFdBQVc7NkJBQ1o7eUJBQ0Y7d0JBQ0QsRUFBRTt3QkFDRjs0QkFDRSxRQUFRLEVBQUUsT0FBTzs0QkFDakIsUUFBUSxFQUFFLGdCQUFnQjt5QkFDM0I7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw0QkFBNEIsQ0FBQyxJQUFVO1lBQ3JDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsZUFBZSxDQUFDLEtBQUssRUFBRTtnQkFDckIsYUFBYSxFQUFFLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxpQkFBaUI7YUFDeEQsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtnQkFDL0QsUUFBUSxFQUFFO29CQUNSLEVBQUU7b0JBQ0Y7d0JBQ0UsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLGVBQWUsRUFBRTtvQ0FDZixXQUFXLEVBQUUsbUJBQW1CO2lDQUNqQzs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELGlDQUFpQyxDQUFDLElBQVU7WUFDMUMsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixlQUFlLENBQUMsS0FBSyxFQUFFO2dCQUNyQixZQUFZLEVBQUU7b0JBQ1osU0FBUyxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUU7b0JBQ2xDLFNBQVMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLGVBQVEsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7aUJBQ2xEO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtnQkFDL0QsUUFBUSxFQUFFO29CQUNSLEVBQUU7b0JBQ0Y7d0JBQ0UsU0FBUyxFQUFFOzRCQUNUO2dDQUNFLGVBQWUsRUFBRTtvQ0FDZixjQUFjLEVBQUUsd0JBQXdCO2lDQUN6Qzs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELHFEQUFxRCxDQUFDLElBQVU7WUFDOUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixlQUFlLENBQUMsS0FBSyxFQUFFO2dCQUNyQixTQUFTLEVBQUUsUUFBUTthQUNwQixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDZCQUE2QixFQUFFO2dCQUMvRCxRQUFRLEVBQUU7b0JBQ1IsRUFBRTtvQkFDRjt3QkFDRSxTQUFTLEVBQUU7NEJBQ1Q7Z0NBQ0UsZUFBZSxFQUFFO29DQUNmLFdBQVcsRUFBRSxRQUFRO2lDQUN0Qjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlTGlrZSB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZSc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0IHsgRHVyYXRpb24sIFNlY3JldFZhbHVlLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCAqIGFzIGNwYWN0aW9ucyBmcm9tICcuLi8uLi9saWInO1xuXG4vLyB0c2xpbnQ6ZGlzYWJsZTpvYmplY3QtbGl0ZXJhbC1rZXktcXVvdGVzXG5cbmV4cG9ydCA9IHtcbiAgJ1MzIERlcGxveSBBY3Rpb24nOiB7XG4gICAgJ2J5IGRlZmF1bHQgZXh0cmFjdCBhcnRpZmFjdHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBtaW5pbWFsUGlwZWxpbmUoc3RhY2spO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICAgJ1N0YWdlcyc6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnTmFtZSc6ICdTb3VyY2UnLFxuICAgICAgICAgICAgJ0FjdGlvbnMnOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnTmFtZSc6ICdTb3VyY2UnLFxuICAgICAgICAgICAgICAgICdBY3Rpb25UeXBlSWQnOiB7XG4gICAgICAgICAgICAgICAgICAnQ2F0ZWdvcnknOiAnU291cmNlJyxcbiAgICAgICAgICAgICAgICAgICdPd25lcic6ICdUaGlyZFBhcnR5JyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgICdOYW1lJzogJ0RlcGxveScsXG4gICAgICAgICAgICAnQWN0aW9ucyc6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdBY3Rpb25UeXBlSWQnOiB7XG4gICAgICAgICAgICAgICAgICAnQ2F0ZWdvcnknOiAnRGVwbG95JyxcbiAgICAgICAgICAgICAgICAgICdQcm92aWRlcic6ICdTMycsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAnQ29uZmlndXJhdGlvbic6IHtcbiAgICAgICAgICAgICAgICAgICdFeHRyYWN0JzogJ3RydWUnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgJ05hbWUnOiAnQ29weUZpbGVzJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdncmFudCB0aGUgcGlwZWxpbmUgY29ycmVjdCBhY2Nlc3MgdG8gdGhlIHRhcmdldCBidWNrZXQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBtaW5pbWFsUGlwZWxpbmUoc3RhY2spO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICAgICdQb2xpY3lEb2N1bWVudCc6IHtcbiAgICAgICAgICAnU3RhdGVtZW50JzogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnRWZmZWN0JzogJ0FsbG93JyxcbiAgICAgICAgICAgICAgJ0FjdGlvbic6IFtcbiAgICAgICAgICAgICAgICAnczM6R2V0T2JqZWN0KicsXG4gICAgICAgICAgICAgICAgJ3MzOkdldEJ1Y2tldConLFxuICAgICAgICAgICAgICAgICdzMzpMaXN0KicsXG4gICAgICAgICAgICAgICAgJ3MzOkRlbGV0ZU9iamVjdConLFxuICAgICAgICAgICAgICAgICdzMzpQdXRPYmplY3QqJyxcbiAgICAgICAgICAgICAgICAnczM6QWJvcnQqJyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7fSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0VmZmVjdCc6ICdBbGxvdycsXG4gICAgICAgICAgICAgICdBY3Rpb24nOiAnc3RzOkFzc3VtZVJvbGUnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2tlYmFiLWNhc2UgQ2FubmVkQUNMIHZhbHVlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgbWluaW1hbFBpcGVsaW5lKHN0YWNrLCB7XG4gICAgICAgIGFjY2Vzc0NvbnRyb2w6IHMzLkJ1Y2tldEFjY2Vzc0NvbnRyb2wuUFVCTElDX1JFQURfV1JJVEUsXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVQaXBlbGluZTo6UGlwZWxpbmUnLCB7XG4gICAgICAgICdTdGFnZXMnOiBbXG4gICAgICAgICAge30sXG4gICAgICAgICAge1xuICAgICAgICAgICAgJ0FjdGlvbnMnOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnQ29uZmlndXJhdGlvbic6IHtcbiAgICAgICAgICAgICAgICAgICdDYW5uZWRBQ0wnOiAncHVibGljLXJlYWQtd3JpdGUnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnYWxsb3cgY3VzdG9taXppbmcgY2FjaGUtY29udHJvbCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICAgIG1pbmltYWxQaXBlbGluZShzdGFjaywge1xuICAgICAgICBjYWNoZUNvbnRyb2w6IFtcbiAgICAgICAgICBjcGFjdGlvbnMuQ2FjaGVDb250cm9sLnNldFB1YmxpYygpLFxuICAgICAgICAgIGNwYWN0aW9ucy5DYWNoZUNvbnRyb2wubWF4QWdlKER1cmF0aW9uLmhvdXJzKDEyKSksXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVQaXBlbGluZTo6UGlwZWxpbmUnLCB7XG4gICAgICAgICdTdGFnZXMnOiBbXG4gICAgICAgICAge30sXG4gICAgICAgICAge1xuICAgICAgICAgICAgJ0FjdGlvbnMnOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnQ29uZmlndXJhdGlvbic6IHtcbiAgICAgICAgICAgICAgICAgICdDYWNoZUNvbnRyb2wnOiAncHVibGljLCBtYXgtYWdlOiA0MzIwMCcsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdhbGxvdyBjdXN0b21pemluZyBvYmplY3RLZXkgKGRlcGxveW1lbnQgcGF0aCBvbiBTMyknKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBtaW5pbWFsUGlwZWxpbmUoc3RhY2ssIHtcbiAgICAgICAgb2JqZWN0S2V5OiAnL2EvYi9jJyxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICAgJ1N0YWdlcyc6IFtcbiAgICAgICAgICB7fSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnQWN0aW9ucyc6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdDb25maWd1cmF0aW9uJzoge1xuICAgICAgICAgICAgICAgICAgJ09iamVjdEtleSc6ICcvYS9iL2MnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG59O1xuXG5pbnRlcmZhY2UgTWluaW1hbFBpcGVsaW5lT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGFjY2Vzc0NvbnRyb2w/OiBzMy5CdWNrZXRBY2Nlc3NDb250cm9sO1xuICByZWFkb25seSBidWNrZXQ/OiBzMy5JQnVja2V0O1xuICByZWFkb25seSBjYWNoZUNvbnRyb2w/OiBjcGFjdGlvbnMuQ2FjaGVDb250cm9sW107XG4gIHJlYWRvbmx5IGV4dHJhY3Q/OiBib29sZWFuO1xuICByZWFkb25seSBvYmplY3RLZXk/OiBzdHJpbmc7XG59XG5cbmZ1bmN0aW9uIG1pbmltYWxQaXBlbGluZShzdGFjazogU3RhY2ssIG9wdGlvbnM6IE1pbmltYWxQaXBlbGluZU9wdGlvbnMgPSB7fSk6IGNvZGVwaXBlbGluZS5JU3RhZ2Uge1xuICBjb25zdCBzb3VyY2VPdXRwdXQgPSBuZXcgY29kZXBpcGVsaW5lLkFydGlmYWN0KCk7XG4gIGNvbnN0IHNvdXJjZUFjdGlvbiA9IG5ldyBjcGFjdGlvbnMuR2l0SHViU291cmNlQWN0aW9uKHtcbiAgICBhY3Rpb25OYW1lOiAnU291cmNlJyxcbiAgICBvd25lcjogJ2F3cycsXG4gICAgcmVwbzogJ2F3cy1jZGsnLFxuICAgIG91dHB1dDogc291cmNlT3V0cHV0LFxuICAgIG9hdXRoVG9rZW46IFNlY3JldFZhbHVlLnBsYWluVGV4dCgnc2VjcmV0JyksXG4gIH0pO1xuXG4gIGNvbnN0IHBpcGVsaW5lID0gbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZShzdGFjaywgJ015UGlwZWxpbmUnLCB7XG4gICAgc3RhZ2VzOiBbXG4gICAgICB7XG4gICAgICAgIHN0YWdlTmFtZTogJ1NvdXJjZScsXG4gICAgICAgIGFjdGlvbnM6IFsgc291cmNlQWN0aW9uIF0sXG4gICAgICB9LFxuICAgIF0sXG4gIH0pO1xuXG4gIGNvbnN0IGRlcGxveW1lbnRTdGFnZSA9IHBpcGVsaW5lLmFkZFN0YWdlKHtcbiAgICBzdGFnZU5hbWU6ICdEZXBsb3knLFxuICAgIGFjdGlvbnM6IFtcbiAgICAgIG5ldyBjcGFjdGlvbnMuUzNEZXBsb3lBY3Rpb24oe1xuICAgICAgICBhY2Nlc3NDb250cm9sOiBvcHRpb25zLmFjY2Vzc0NvbnRyb2wsXG4gICAgICAgIGFjdGlvbk5hbWU6ICdDb3B5RmlsZXMnLFxuICAgICAgICBidWNrZXQ6IG9wdGlvbnMuYnVja2V0IHx8IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUJ1Y2tldCcpLFxuICAgICAgICBjYWNoZUNvbnRyb2w6IG9wdGlvbnMuY2FjaGVDb250cm9sLFxuICAgICAgICBleHRyYWN0OiBvcHRpb25zLmV4dHJhY3QsXG4gICAgICAgIGlucHV0OiBzb3VyY2VPdXRwdXQsXG4gICAgICAgIG9iamVjdEtleTogb3B0aW9ucy5vYmplY3RLZXksXG4gICAgICB9KSxcbiAgICBdLFxuICB9KTtcblxuICByZXR1cm4gZGVwbG95bWVudFN0YWdlO1xufVxuIl19