"use strict";
const assert_1 = require("@aws-cdk/assert");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cpactions = require("../../lib");
function minimalPipeline(stack, options = {}) {
    const sourceOutput = new codepipeline.Artifact();
    const pipeline = new codepipeline.Pipeline(stack, 'MyPipeline');
    const sourceStage = pipeline.addStage({
        stageName: 'Source',
        actions: [
            new cpactions.S3SourceAction({
                actionName: 'Source',
                bucket: options.bucket || new s3.Bucket(stack, 'MyBucket'),
                bucketKey: options.bucketKey || 'some/path/to',
                output: sourceOutput,
                trigger: options.trigger,
            }),
        ],
    });
    pipeline.addStage({
        stageName: 'Build',
        actions: [
            new cpactions.CodeBuildAction({
                actionName: 'Build',
                project: new codebuild.PipelineProject(stack, 'MyProject'),
                input: sourceOutput,
            }),
        ],
    });
    return sourceStage;
}
module.exports = {
    'S3 Source Action': {
        'by default polls for source changes and does not use Events'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, undefined);
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Actions': [
                            {
                                'Configuration': {},
                            },
                        ],
                    },
                    {},
                ],
            }));
            assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::Events::Rule')));
            test.done();
        },
        'does not poll for source changes and uses Events for S3Trigger.EVENTS'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, { trigger: cpactions.S3Trigger.EVENTS });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'PollForSourceChanges': false,
                                },
                            },
                        ],
                    },
                    {},
                ],
            }));
            assert_1.expect(stack).to(assert_1.countResources('AWS::Events::Rule', 1));
            test.done();
        },
        'polls for source changes and does not use Events for S3Trigger.POLL'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, { trigger: cpactions.S3Trigger.POLL });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'PollForSourceChanges': true,
                                },
                            },
                        ],
                    },
                    {},
                ],
            }));
            assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::Events::Rule')));
            test.done();
        },
        'does not poll for source changes and does not use Events for S3Trigger.NONE'(test) {
            const stack = new core_1.Stack();
            minimalPipeline(stack, { trigger: cpactions.S3Trigger.NONE });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Actions': [
                            {
                                'Configuration': {
                                    'PollForSourceChanges': false,
                                },
                            },
                        ],
                    },
                    {},
                ],
            }));
            assert_1.expect(stack).to(assert_1.not(assert_1.haveResourceLike('AWS::Events::Rule')));
            test.done();
        },
        'does not allow passing an empty string for the bucketKey property'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new cpactions.S3SourceAction({
                    actionName: 'Source',
                    bucket: new s3.Bucket(stack, 'MyBucket'),
                    bucketKey: '',
                    output: new codepipeline.Artifact(),
                });
            }, /Property bucketKey cannot be an empty string/);
            test.done();
        },
        'allows using the same bucket with events trigger mutliple times with different bucket paths'(test) {
            const stack = new core_1.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const sourceStage = minimalPipeline(stack, {
                bucket,
                bucketKey: 'my/path',
                trigger: cpactions.S3Trigger.EVENTS,
            });
            sourceStage.addAction(new cpactions.S3SourceAction({
                actionName: 'Source2',
                bucket,
                bucketKey: 'my/other/path',
                trigger: cpactions.S3Trigger.EVENTS,
                output: new codepipeline.Artifact(),
            }));
            test.done();
        },
        'throws an error if the same bucket and path with trigger = Events are added to the same pipeline twice'(test) {
            const stack = new core_1.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const sourceStage = minimalPipeline(stack, {
                bucket,
                bucketKey: 'my/path',
                trigger: cpactions.S3Trigger.EVENTS,
            });
            sourceStage.addAction(new cpactions.S3SourceAction({
                actionName: 'Source2',
                bucket,
                bucketKey: 'my/other/path',
                trigger: cpactions.S3Trigger.EVENTS,
                output: new codepipeline.Artifact(),
            }));
            const duplicateBucketAndPath = new cpactions.S3SourceAction({
                actionName: 'Source3',
                bucket,
                bucketKey: 'my/other/path',
                trigger: cpactions.S3Trigger.EVENTS,
                output: new codepipeline.Artifact(),
            });
            test.throws(() => {
                sourceStage.addAction(duplicateBucketAndPath);
            }, /S3 source action with path 'my\/other\/path' is already present in the pipeline for this source bucket/);
            test.done();
        },
        'exposes variables for other actions to consume'(test) {
            const stack = new core_1.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const s3SourceAction = new cpactions.S3SourceAction({
                actionName: 'Source',
                output: sourceOutput,
                bucket: new s3.Bucket(stack, 'Bucket'),
                bucketKey: 'key.zip',
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [s3SourceAction],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new cpactions.CodeBuildAction({
                                actionName: 'Build',
                                project: new codebuild.PipelineProject(stack, 'MyProject'),
                                input: sourceOutput,
                                environmentVariables: {
                                    VersionId: { value: s3SourceAction.variables.versionId },
                                },
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source',
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'Build',
                                'Configuration': {
                                    'EnvironmentVariables': '[{"name":"VersionId","type":"PLAINTEXT","value":"#{Source_Source_NS.VersionId}"}]',
                                },
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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