# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['meilisearch_python_async', 'meilisearch_python_async.models']

package_data = \
{'': ['*']}

install_requires = \
['aiofiles>=0.7.0,<0.8.0',
 'camel-converter>=1.0.0,<2.0.0',
 'httpx>=0.17,<0.20',
 'pydantic>=1.8,<2.0']

setup_kwargs = {
    'name': 'meilisearch-python-async',
    'version': '0.12.0',
    'description': 'A Python async client for the MeiliSearch API',
    'long_description': '# MeiliSearch Python Async\n\n[![Tests Status](https://github.com/sanders41/meilisearch-python-async/workflows/Testing/badge.svg?branch=main&event=push)](https://github.com/sanders41/meilisearch-python-async/actions?query=workflow%3ATesting+branch%3Amain+event%3Apush)\n[![pre-commit.ci status](https://results.pre-commit.ci/badge/github/sanders41/meilisearch-python-async/main.svg)](https://results.pre-commit.ci/latest/github/sanders41/meilisearch-python-async/main)\n[![Coverage](https://codecov.io/github/sanders41/meilisearch-python-async/coverage.svg?branch=main)](https://codecov.io/gh/sanders41/meilisearch-python-async)\n[![PyPI version](https://badge.fury.io/py/meilisearch-python-async.svg)](https://badge.fury.io/py/meilisearch-python-async)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/meilisearch-python-async?color=5cc141)](https://github.com/sanders41/meilisearch-python-async)\n\nMeilisearch Python Async is a Python async client for the [MeiliSearch](https://github.com/meilisearch/MeiliSearch) API. MeiliSearch also has an official [Python client](https://github.com/meilisearch/meilisearch-python).\n\nWhich of the two clients to use comes down to your particular use case. The purpose for this async client is to allow for non-blocking calls when working in async frameworks such as [FastAPI](https://fastapi.tiangolo.com/), or if your own code base you are working in is async. If this does not match your use case then the official client will be a better choice.\n\nFor the most part this client mirrors the functionality of the official client and the same [documenation](https://docs.meilisearch.com/reference/) will apply. There are are few exceptions to this to be aware of:\n\n1. The async client runs in a context manager. This means to create a client, instead of:\n\n    ```py\n    client = Client("http://127.0.0.1:7700", "masterKey")\n    ```\n\n    In the async client it would be:\n\n    ```py\n    async with Client("http://127.0.0.1:7700", "masterKey") as client:\n        ...\n    ```\n\n2. Because this client is async you need to await the calls. For example adding documents with the official verison of the client would be:\n\n    ```py\n    index.add_documents(documents)\n    ```\n\n    In the async client it would be:\n\n    ```py\n    await index.add_documents(documents)\n    ```\n\n3. The async client uses [Pydantic](https://pydantic-docs.helpmanual.io/) to serialize/deserialize the JSON from MeiliSearch into Python objects wherever possible, and in the process converts the camelCaseNames from JSON into more Pythonic snake_case_names. The official client instead uses dictionaries to store the return values in most cases.\n\nIn some instances it is not possible to return the data as an object because the structure will be dependant on your particular dataset and can\'t\nbe known ahead of time. In these instances you can either work with the data in the dictionary that is returned, or because you will know the\nstructure you can generate your own Pydantic models.\n\nAs an example, if you want to get a movie from the [small movies example](https://github.com/sanders41/meilisearch-python-async/blob/main/datasets/small_movies.json) you could put the results into an object with the following\n\n```py\nfrom datetime import datetime\nfrom typing import Optional\n\nfrom meilisearch_python_async import Client\nfrom meilisearch_python_async.models import CamelBase\n\n\n# Inheriting from CamelBase will allow your class to automatically convert\n# variables returned from the server in camelCase into snake_case. It will\n# also make it a Pydantic Model.\nclass Movie(CamelBase):\n    id: int\n    title: str\n    poster: str\n    overview: str\n    release_date: datetime\n    genre: Optional[str] = None\n\n\nasync with Client("http://127.0.0.1:7700", "masterKey") as client:\n    index = client.index("movies")\n    movie_dict = await index.get_document(287947)\n    movie = Movie(**movie_dict)\n```\n\nAnd then the movie variable would contain the movie object with the following information\n\n```py\nMovie(\n    id = 287947,\n    title = "Shazam!",\n    poster = "https://image.tmdb.org/t/p/w1280/xnopI5Xtky18MPhK40cZAGAOVeV.jpg",\n    overview = "A boy is given the ability to become an adult superhero in times of need with a single magic word.",\n    release_date = datetime.datetime(2019, 3, 23, 0, 0, tzinfo=datetime.timezone.utc),\n    genre = "action",\n)\n```\n\nBy inheriting from CamelBase, or any of the other [provided models](https://github.com/sanders41/meilisearch-python-async/tree/main/meilisearch_python_async/models)\nyou will be inheriting Pydantic models and therefore have access to the funcitonality Pydantic provides\nsuch as [validators](https://pydantic-docs.helpmanual.io/usage/validators/) and [Fields](https://pydantic-docs.helpmanual.io/usage/model_config/#alias-precedence). Pydantic will also automatically deserialized the data into the correct data type based on the type hint provided.\n\n## Installation\n\nUsing a virtual environmnet is recommended for installing this package. Once the virtual environment is created and activated install the package with:\n\n```sh\npip install meilisearch-python-async\n```\n\n## Run MeiliSearch\n\nThere are several ways to [run MeiliSearch](https://docs.meilisearch.com/reference/features/installation.html#download-and-launch).\nPick the one that works best for your use case and then start the server.\n\nAs as example to use Docker:\n\n```sh\ndocker pull getmeili/meilisearch:latest\ndocker run -it --rm -p 7700:7700 getmeili/meilisearch:latest ./meilisearch --master-key=masterKey\n```\n\n## Useage\n\n### Add Documents\n\n* Note: `client.index("books") creates an instance of an Index object but does not make a network call to send the data yet so it does not need to be awaited.\n\n```py\nfrom meilisearch_python_async import Client\n\nasync with Client(\'http://127.0.0.1:7700\', \'masterKey\') as client:\n    index = client.index("books")\n\n    documents = [\n        {"id": 1, "title": "Ready Player One"},\n        {"id": 42, "title": "The Hitchhiker\'s Guide to the Galaxy"},\n    ]\n\n    await index.add_documents(documents)\n```\n\nThe server will return an update id that can be used to [get the status](https://docs.meilisearch.com/reference/api/updates.html#get-an-update-status)\nof the updates. To do this you would save the result response from adding the documets to a variable,\nthis will be a UpdateId object, and use it to check the status of the updates.\n\n```py\nupdate = await index.add_documents(documents)\nstatus = await client.index(\'books\').get_update_status(update.update_id)\n```\n\n### Add Documents In Batches\n\nSplitting documents into batches can be useful with large dataset because it reduces the RAM usage\nduring indexing.\n\n```py\nfrom meilisearch_python_async import Client\n\nasync with Client(\'http://127.0.0.1:7700\', \'masterKey\') as client:\n    index = client.index("books")\n\n    documents = [\n        {"id": 1, "title": "Ready Player One"},\n        {"id": 42, "title": "The Hitchhiker\'s Guide to the Galaxy"},\n        ...\n    ]\n\n    await index.add_documents_in_batches(documents, batch_size=100)\n```\n\nThe server will return a list of update ids that can be used to [get the status](https://docs.meilisearch.com/reference/api/updates.html#get-an-update-status)\nof the updates. To do this you would save the result response from adding the documets to a variable,\nthis will be a list of UpdateId objects, and use it to check the status of the updates.\n\n```py\nupdates = await index.add_documents_in_batches(documents, batch_size=20)\nfor update in updates:\n    status = await client.index(\'books\').get_update_status(update.update_id)\n```\n\n### Basic Searching\n\n```py\nsearch_result = await index.search("ready player")\n```\n\n### Base Search Results: SearchResults object with values\n\n```py\nSearchResults(\n    hits = [\n        {\n            "id": 1,\n            "title": "Ready Player One",\n        },\n    ],\n    offset = 0,\n    limit = 20,\n    nb_hits = 1,\n    exhaustive_nb_hits = bool,\n    facets_distributionn = None,\n    processing_time_ms = 1,\n    query = "ready player",\n)\n```\n\n### Custom Search\n\nInformation about the parameters can be found in the [search parameters](https://docs.meilisearch.com/reference/features/search_parameters.html) section of the documentation.\n\n```py\nindex.search(\n    "guide",\n    attributes_to_highlight=["title"],\n    filters="book_id > 10"\n)\n```\n\n### Custom Search Results: SearchResults object with values\n\n```py\nSearchResults(\n    hits = [\n        {\n            "id": 42,\n            "title": "The Hitchhiker\'s Guide to the Galaxy",\n            "_formatted": {\n                "id": 42,\n                "title": "The Hitchhiker\'s Guide to the <em>Galaxy</em>"\n            }\n        },\n    ],\n    offset = 0,\n    limit = 20,\n    nb_hits = 1,\n    exhaustive_nb_hits = bool,\n    facets_distributionn = None,\n    processing_time_ms = 5,\n    query = "galaxy",\n)\n```\n\n### The following methods are unique to this client and are not currently available in the official client, or in the MeiliSearch documentation\n\n* add_documents_from_file:\n\n  Add documents to an index from a json file. The file must have a .json extension. The file path\n  can be passed either as a string or as a Path object.\n\n  ```py\n  index = test_client.index("movies")\n  response = await index.add_documents_from_file("/path/to/file.json")\n  ```\n\n* update_documents_from_file:\n\n  Update documents in an index from a json file. The file must have a .json extension. The file path\n  can be passed either as a string or as a Path object.\n\n  ```py\n  index = test_client.index("movies")\n  response = await index.update_documents_from_file("/path/to/file.json")\n  ```\n\n* add_documents_auto_batch:\n\n  Automatically split document into batches when adding documents. The auto batcher puts as may\n  documents as possible into the batch while fitting under the maximum payload size (default is 100MB)\n\n  ```py\n  index = test_client.index("movies")\n  response = await index.add_documents_auto_batch(documents)\n  ```\n\n* add_documents_from_file_auto_batch:\n\n  Automatically split document into batches when adding documents from a file. The auto batcher puts\n  as may documents as possible into the batch while fitting under the maximum payload size (default is 100MB)\n\n  ```py\n  index = test_client.index("movies")\n  response = await index.add_documents_from_file_auto_batch("/path/to/file.json")\n  ```\n\n* update_documents_auto_batch:\n\n  Automatically split document into batches when updating documents. The auto batcher puts as may\n  documents as possible into the batch while fitting under the maximum payload size (default is 100MB)\n\n  ```py\n  index = test_client.index("movies")\n  response = await index.update_documents_auto_batch(documents)\n  ```\n\n* update_documents_from_file_auto_batch:\n\n  Automatically split document into batches when updating documents from a file. The auto batcher\n  puts as may documents as possible into the batch while fitting under the maximum payload size (default is 100MB)\n\n  ```py\n  index = test_client.index("movies")\n  response = await index.update_documents_from_file_auto_batch("/path/to/file.json")\n  ```\n\n## Compatibility with MeiliSearch\n\nThis package only guarantees the compatibility with [version v0.21.0 of MeiliSearch](https://github.com/meilisearch/MeiliSearch/releases/tag/v0.21.0).\n\n## Learn More\n\nFor more see the [API Reference](https://docs.meilisearch.com/reference/api/) in the MeiliSearch documentation. Keep in mind you will need to await the examples shown in the documentation, and that you will be getting python objects instead of JSON for you results.\n\n## Contributing\n\nContributions to this project are welcome. If you are interesting in contributing please see our [contributing guide](CONTRIBUTING.md)\n',
    'author': 'Paul Sanders',
    'author_email': 'psanders1@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/sanders41/meilisearch-python-async',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
