"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnvironmentVariablesService = void 0;
const _1 = require(".");
/**
 * Class EnvironmentVariablesService
 *
 * This class is used to return environment variables that are available in the runtime of
 * the current Lambda invocation.
 * These variables can be a mix of runtime environment variables set by AWS and
 * variables that can be set by the developer additionally.
 *
 * @class
 * @extends {ConfigService}
 * @see https://docs.aws.amazon.com/lambda/latest/dg/configuration-envvars.html#configuration-envvars-runtime
 * @see https://awslabs.github.io/aws-lambda-powertools-typescript/latest/#environment-variables
 */
class EnvironmentVariablesService extends _1.ConfigService {
    constructor() {
        super(...arguments);
        /**
         * @see https://awslabs.github.io/aws-lambda-powertools-typescript/latest/#environment-variables
         * @protected
         */
        this.serviceNameVariable = 'POWERTOOLS_SERVICE_NAME';
        // Reserved environment variables
        this.xRayTraceIdVariable = '_X_AMZN_TRACE_ID';
    }
    /**
     * It returns the value of an environment variable that has given name.
     *
     * @param {string} name
     * @returns {string}
     */
    get(name) {
        var _a;
        return ((_a = process.env[name]) === null || _a === void 0 ? void 0 : _a.trim()) || '';
    }
    /**
     * It returns the value of the POWERTOOLS_SERVICE_NAME environment variable.
     *
     * @returns {string}
     */
    getServiceName() {
        return this.get(this.serviceNameVariable);
    }
    /**
     * It returns the value of the _X_AMZN_TRACE_ID environment variable.
     *
     * The AWS X-Ray Trace data available in the environment variable has this format:
     * `Root=1-5759e988-bd862e3fe1be46a994272793;Parent=557abcec3ee5a047;Sampled=1`,
     *
     * The actual Trace ID is: `1-5759e988-bd862e3fe1be46a994272793`.
     *
     * @returns {string}
     */
    getXrayTraceId() {
        const xRayTraceId = this.get(this.xRayTraceIdVariable);
        if (xRayTraceId === '')
            return undefined;
        return xRayTraceId.split(';')[0].replace('Root=', '');
    }
}
exports.EnvironmentVariablesService = EnvironmentVariablesService;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiRW52aXJvbm1lbnRWYXJpYWJsZXNTZXJ2aWNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2NvbmZpZy9FbnZpcm9ubWVudFZhcmlhYmxlc1NlcnZpY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0JBQWtDO0FBRWxDOzs7Ozs7Ozs7Ozs7R0FZRztBQUNILE1BQU0sMkJBQTRCLFNBQVEsZ0JBQWE7SUFBdkQ7O1FBRUU7OztXQUdHO1FBQ08sd0JBQW1CLEdBQUcseUJBQXlCLENBQUM7UUFDMUQsaUNBQWlDO1FBQ3pCLHdCQUFtQixHQUFHLGtCQUFrQixDQUFDO0lBdUNuRCxDQUFDO0lBckNDOzs7OztPQUtHO0lBQ0ksR0FBRyxDQUFDLElBQVk7O1FBQ3JCLE9BQU8sQ0FBQSxNQUFBLE9BQU8sQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLDBDQUFFLElBQUksRUFBRSxLQUFJLEVBQUUsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxjQUFjO1FBQ25CLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFFdkQsSUFBSSxXQUFXLEtBQUssRUFBRTtZQUFFLE9BQU8sU0FBUyxDQUFDO1FBRXpDLE9BQU8sV0FBVyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ3hELENBQUM7Q0FFRjtBQUdDLGtFQUEyQiJ9