"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Logger = void 0;
const console_1 = require("console");
const commons_1 = require("@aws-lambda-powertools/commons");
const formatter_1 = require("./formatter");
const log_1 = require("./log");
const lodash_merge_1 = __importDefault(require("lodash.merge"));
const config_1 = require("./config");
/**
 * ## Intro
 * The Logger utility provides an opinionated logger with output structured as JSON.
 *
 * ## Key features
 *  * Capture key fields from Lambda context, cold start and structures logging output as JSON
 *  * Log Lambda context when instructed (disabled by default)
 *  * Log sampling prints all logs for a percentage of invocations (disabled by default)
 *  * Append additional keys to structured log at any point in time
 *
 * ## Usage
 *
 * For more usage examples, see [our documentation](https://awslabs.github.io/aws-lambda-powertools-typescript/latest/core/logger/).
 *
 * ### Basic usage
 *
 * @example
 * ```typescript
 * import { Logger } from '@aws-lambda-powertools/logger';
 *
 * // Logger parameters fetched from the environment variables:
 * const logger = new Logger();
 * ```
 *
 * ### Functions usage with middleware
 *
 * If you use function-based Lambda handlers you can use the [injectLambdaContext()](#injectLambdaContext)
 * middy middleware to automatically add context to your Lambda logs.
 *
 * @example
 * ```typescript
 * import { Logger, injectLambdaContext } from '@aws-lambda-powertools/logger';
 * import middy from '@middy/core';
 *
 * const logger = new Logger();
 *
 * const lambdaHandler = async (_event: any, _context: any) => {
 *     logger.info('This is an INFO log with some context');
 * };
 *
 * export const handler = middy(lambdaHandler).use(injectLambdaContext(logger));
 * ```
 *
 * ### Object oriented usage with decorators
 *
 * If instead you use TypeScript classes to wrap your Lambda handler you can use the [@logger.injectLambdaContext()](./_aws_lambda_powertools_logger.Logger.html#injectLambdaContext) decorator.
 *
 * @example
 * ```typescript
 * import { Logger } from '@aws-lambda-powertools/logger';
 * import { LambdaInterface } from '@aws-lambda-powertools/commons';
 *
 * const logger = new Logger();
 *
 * class Lambda implements LambdaInterface {
 *
 *   // FYI: Decorator might not render properly in VSCode mouse over due to https://github.com/microsoft/TypeScript/issues/47679 and might show as *@logger* instead of `@logger.injectLambdaContext`
 *
 *     // Decorate your handler class method
 *     @logger.injectLambdaContext()
 *     public async handler(_event: any, _context: any): Promise<void> {
 *         logger.info('This is an INFO log with some context');
 *     }
 * }
 *
 * const handlerClass = new Lambda();
 * export const handler = handlerClass.handler.bind(handlerClass);
 * ```
 *
 * ### Functions usage with manual instrumentation
 *
 * If you prefer to manually instrument your Lambda handler you can use the methods in the Logger class directly.
 *
 * @example
 * ```typescript
 * import { Logger } from '@aws-lambda-powertools/logger';
 *
 * const logger = new Logger();
 *
 * export const handler = async (_event, context) => {
 *     logger.addContext(context);
 *     logger.info('This is an INFO log with some context');
 * };
 * ```
 *
 * @class
 * @implements {ClassThatLogs}
 * @see https://awslabs.github.io/aws-lambda-powertools-typescript/latest/core/logger/
 */
class Logger extends commons_1.Utility {
    /**
     * It initializes the Logger class with an optional set of options (settings).
     * *
     * @param {ConstructorOptions} options
     */
    constructor(options = {}) {
        super();
        this.logEvent = false;
        this.logIndentation = 0 /* COMPACT */;
        this.logLevelThresholds = {
            DEBUG: 8,
            INFO: 12,
            WARN: 16,
            ERROR: 20,
        };
        this.logsSampled = false;
        this.persistentLogAttributes = {};
        this.powertoolLogData = {};
        this.setOptions(options);
    }
    /**
     * It adds the current Lambda function's invocation context data to the powertoolLogData property of the instance.
     * This context data will be part of all printed log items.
     *
     * @param {Context} context
     * @returns {void}
     */
    addContext(context) {
        const lambdaContext = {
            invokedFunctionArn: context.invokedFunctionArn,
            coldStart: this.getColdStart(),
            awsRequestId: context.awsRequestId,
            memoryLimitInMB: Number(context.memoryLimitInMB),
            functionName: context.functionName,
            functionVersion: context.functionVersion,
        };
        this.addToPowertoolLogData({
            lambdaContext,
        });
    }
    /**
     * It adds the given attributes (key-value pairs) to all log items generated by this Logger instance.
     *
     * @param {LogAttributes} attributes
     * @returns {void}
     */
    addPersistentLogAttributes(attributes) {
        (0, lodash_merge_1.default)(this.persistentLogAttributes, attributes);
    }
    /**
     * Alias for addPersistentLogAttributes.
     *
     * @param {LogAttributes} attributes
     * @returns {void}
     */
    appendKeys(attributes) {
        this.addPersistentLogAttributes(attributes);
    }
    /**
     * It creates a separate Logger instance, identical to the current one
     * It's possible to overwrite the new instance options by passing them.
     *
     * @param {ConstructorOptions} options
     * @returns {Logger}
     */
    createChild(options = {}) {
        const parentsPowertoolsLogData = this.getPowertoolLogData();
        const childLogger = new Logger((0, lodash_merge_1.default)({}, parentsPowertoolsLogData, options));
        const parentsPersistentLogAttributes = this.getPersistentLogAttributes();
        childLogger.addPersistentLogAttributes(parentsPersistentLogAttributes);
        if (parentsPowertoolsLogData.lambdaContext) {
            childLogger.addContext(parentsPowertoolsLogData.lambdaContext);
        }
        return childLogger;
    }
    /**
     * It prints a log item with level DEBUG.
     *
     * @param {LogItemMessage} input
     * @param {Error | LogAttributes | string} extraInput
     * @returns {void}
     */
    debug(input, ...extraInput) {
        this.processLogItem('DEBUG', input, extraInput);
    }
    /**
     * It prints a log item with level ERROR.
     *
     * @param {LogItemMessage} input
     * @param {Error | LogAttributes | string} extraInput
     * @returns {void}
     */
    error(input, ...extraInput) {
        this.processLogItem('ERROR', input, extraInput);
    }
    /**
     * It returns a boolean value. True means that the Lambda invocation events
     * are printed in the logs.
     *
     * @returns {boolean}
     */
    getLogEvent() {
        return this.logEvent;
    }
    /**
     * It returns a boolean value, if true all the logs will be printed.
     *
     * @returns {boolean}
     */
    getLogsSampled() {
        return this.logsSampled;
    }
    /**
     * It returns the persistent log attributes, which are the attributes
     * that will be logged in all log items.
     *
     * @private
     * @returns {LogAttributes}
     */
    getPersistentLogAttributes() {
        return this.persistentLogAttributes;
    }
    /**
     * It prints a log item with level INFO.
     *
     * @param {LogItemMessage} input
     * @param {Error | LogAttributes | string} extraInput
     * @returns {void}
     */
    info(input, ...extraInput) {
        this.processLogItem('INFO', input, extraInput);
    }
    /**
     * Method decorator that adds the current Lambda function context as extra
     * information in all log items.
     *
     * The decorator can be used only when attached to a Lambda function handler which
     * is written as method of a class, and should be declared just before the handler declaration.
     *
     * Note: Currently TypeScript only supports decorators on classes and methods. If you are using the
     * function syntax, you should use the middleware instead.
     *
     * @example
     * ```typescript
     * import { Logger } from '@aws-lambda-powertools/logger';
     * import { LambdaInterface } from '@aws-lambda-powertools/commons';
     *
     * const logger = new Logger();
     *
     * class Lambda implements LambdaInterface {
     *     // Decorate your handler class method
     *     @logger.injectLambdaContext()
     *     public async handler(_event: any, _context: any): Promise<void> {
     *         logger.info('This is an INFO log with some context');
     *     }
     * }
     *
     * const handlerClass = new Lambda();
     * export const handler = handlerClass.handler.bind(handlerClass);
     * ```
     *
     * @see https://www.typescriptlang.org/docs/handbook/decorators.html#method-decorators
     * @returns {HandlerMethodDecorator}
     */
    injectLambdaContext(options) {
        return (_target, _propertyKey, descriptor) => {
            /**
             * The descriptor.value is the method this decorator decorates, it cannot be undefined.
             */
            const originalMethod = descriptor.value;
            // eslint-disable-next-line @typescript-eslint/no-this-alias
            const loggerRef = this;
            // Use a function() {} instead of an () => {} arrow function so that we can
            // access `myClass` as `this` in a decorated `myClass.myMethod()`.
            descriptor.value = (async function (event, context, callback) {
                let initialPersistentAttributes = {};
                if (options && options.clearState === true) {
                    initialPersistentAttributes = { ...loggerRef.getPersistentLogAttributes() };
                }
                Logger.injectLambdaContextBefore(loggerRef, event, context, options);
                /* eslint-disable  @typescript-eslint/no-non-null-assertion */
                let result;
                try {
                    result = await originalMethod.apply(this, [event, context, callback]);
                }
                catch (error) {
                    throw error;
                }
                finally {
                    Logger.injectLambdaContextAfterOrOnError(loggerRef, initialPersistentAttributes, options);
                }
                return result;
            });
        };
    }
    static injectLambdaContextAfterOrOnError(logger, initialPersistentAttributes, options) {
        if (options && options.clearState === true) {
            logger.setPersistentLogAttributes(initialPersistentAttributes);
        }
    }
    static injectLambdaContextBefore(logger, event, context, options) {
        logger.addContext(context);
        let shouldLogEvent = undefined;
        if (options && options.hasOwnProperty('logEvent')) {
            shouldLogEvent = options.logEvent;
        }
        logger.logEventIfEnabled(event, shouldLogEvent);
    }
    /**
     * Logs a Lambda invocation event, if it *should*.
     *
     ** @param {unknown} event
     * @param {boolean} [overwriteValue]
     * @returns {void}
     */
    logEventIfEnabled(event, overwriteValue) {
        if (!this.shouldLogEvent(overwriteValue)) {
            return;
        }
        this.info('Lambda invocation event', { event });
    }
    /**
     * If the sample rate feature is enabled, the calculation that determines whether the logs
     * will actually be printed or not for this invocation is done when the Logger class is
     * initialized.
     * This method will repeat that calculation (with possible different outcome).
     *
     * @returns {void}
     */
    refreshSampleRateCalculation() {
        this.setLogsSampled();
    }
    /**
     * Alias for removePersistentLogAttributes.
     *
     * @param {string[]} keys
     * @returns {void}
     */
    removeKeys(keys) {
        this.removePersistentLogAttributes(keys);
    }
    /**
     * It removes attributes based on provided keys to all log items generated by this Logger instance.
     *
     * @param {string[]} keys
     * @returns {void}
     */
    removePersistentLogAttributes(keys) {
        keys.forEach((key) => {
            if (this.persistentLogAttributes && key in this.persistentLogAttributes) {
                delete this.persistentLogAttributes[key];
            }
        });
    }
    /**
     * It sets the given attributes (key-value pairs) to all log items generated by this Logger instance.
     * Note: this replaces the pre-existing value.
     *
     * @param {LogAttributes} attributes
     * @returns {void}
     */
    setPersistentLogAttributes(attributes) {
        this.persistentLogAttributes = attributes;
    }
    /**
     * It sets the user-provided sample rate value.
     *
     * @param {number} [sampleRateValue]
     * @returns {void}
     */
    setSampleRateValue(sampleRateValue) {
        var _a;
        this.powertoolLogData.sampleRateValue =
            sampleRateValue ||
                ((_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getSampleRateValue()) ||
                this.getEnvVarsService().getSampleRateValue();
    }
    /**
     * It checks whether the current Lambda invocation event should be printed in the logs or not.
     *
     * @private
     * @param {boolean} [overwriteValue]
     * @returns {boolean}
     */
    shouldLogEvent(overwriteValue) {
        if (typeof overwriteValue === 'boolean') {
            return overwriteValue;
        }
        return this.getLogEvent();
    }
    /**
     * It prints a log item with level WARN.
     *
     * @param {LogItemMessage} input
     * @param {Error | LogAttributes | string} extraInput
     * @returns {void}
     */
    warn(input, ...extraInput) {
        this.processLogItem('WARN', input, extraInput);
    }
    /**
     * It stores information that is printed in all log items.
     *
     * @param {Partial<PowertoolLogData>} attributesArray
     * @private
     * @returns {void}
     */
    addToPowertoolLogData(...attributesArray) {
        attributesArray.forEach((attributes) => {
            (0, lodash_merge_1.default)(this.powertoolLogData, attributes);
        });
    }
    /**
     * It processes a particular log item so that it can be printed to stdout:
     * - Merges ephemeral log attributes with persistent log attributes (printed for all logs) and additional info;
     * - Formats all the log attributes;
     *
     * @private
     * @param {LogLevel} logLevel
     * @param {LogItemMessage} input
     * @param {LogItemExtraInput} extraInput
     * @returns {LogItem}
     */
    createAndPopulateLogItem(logLevel, input, extraInput) {
        // TODO: this method's logic is hard to understand, there is an opportunity here to simplify this logic.
        const unformattedBaseAttributes = (0, lodash_merge_1.default)({
            logLevel,
            timestamp: new Date(),
            message: typeof input === 'string' ? input : input.message,
            xRayTraceId: this.envVarsService.getXrayTraceId(),
        }, this.getPowertoolLogData());
        const logItem = new log_1.LogItem({
            baseAttributes: this.getLogFormatter().formatAttributes(unformattedBaseAttributes),
            persistentAttributes: this.getPersistentLogAttributes(),
        });
        // Add ephemeral attributes
        if (typeof input !== 'string') {
            logItem.addAttributes(input);
        }
        extraInput.forEach((item) => {
            const attributes = item instanceof Error ? { error: item } :
                typeof item === 'string' ? { extra: item } :
                    item;
            logItem.addAttributes(attributes);
        });
        return logItem;
    }
    /**
     * It returns the custom config service, an abstraction used to fetch environment variables.
     *
     * @private
     * @returns {ConfigServiceInterface | undefined}
     */
    getCustomConfigService() {
        return this.customConfigService;
    }
    /**
     * It returns the instance of a service that fetches environment variables.
     *
     * @private
     * @returns {EnvironmentVariablesService}
     */
    getEnvVarsService() {
        return this.envVarsService;
    }
    /**
     * It returns the instance of a service that formats the structure of a
     * log item's keys and values in the desired way.
     *
     * @private
     * @returns {LogFormatterInterface}
     */
    getLogFormatter() {
        return this.logFormatter;
    }
    /**
     * It returns the log level set for the Logger instance.
     *
     * @private
     * @returns {LogLevel}
     */
    getLogLevel() {
        return this.logLevel;
    }
    /**
     * It returns information that will be added in all log item by
     * this Logger instance (different from user-provided persistent attributes).
     *
     * @private
     * @returns {LogAttributes}
     */
    getPowertoolLogData() {
        return this.powertoolLogData;
    }
    /**
     * It returns the numeric sample rate value.
     *
     * @private
     * @returns {number}
     */
    getSampleRateValue() {
        if (!this.powertoolLogData.sampleRateValue) {
            this.setSampleRateValue();
        }
        return this.powertoolLogData.sampleRateValue;
    }
    /**
     * It returns true if the provided log level is valid.
     *
     * @param {LogLevel} logLevel
     * @private
     * @returns {boolean}
     */
    isValidLogLevel(logLevel) {
        return typeof logLevel === 'string' && logLevel.toUpperCase() in this.logLevelThresholds;
    }
    /**
     * It prints a given log with given log level.
     *
     * @param {LogLevel} logLevel
     * @param {LogItem} log
     * @private
     */
    printLog(logLevel, log) {
        log.prepareForPrint();
        const consoleMethod = logLevel.toLowerCase();
        this.console[consoleMethod](JSON.stringify(log.getAttributes(), this.removeCircularDependencies(), this.logIndentation));
    }
    /**
     * It prints a given log with given log level.
     *
     * @param {LogLevel} logLevel
     * @param {LogItem} log
     * @private
     */
    processLogItem(logLevel, input, extraInput) {
        if (!this.shouldPrint(logLevel)) {
            return;
        }
        this.printLog(logLevel, this.createAndPopulateLogItem(logLevel, input, extraInput));
    }
    /**
     * When the data added in the log item contains object references,
     * JSON.stringify() doesn't try to solve them and instead throws an error: TypeError: cyclic object value.
     * To mitigate this issue, this method will find and remove all cyclic references.
     *
     * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Errors/Cyclic_object_value
     * @private
     */
    removeCircularDependencies() {
        const references = new WeakSet();
        return (key, value) => {
            let item = value;
            if (item instanceof Error) {
                item = this.getLogFormatter().formatError(item);
            }
            if (typeof item === 'object' && value !== null) {
                if (references.has(item)) {
                    return;
                }
                references.add(item);
            }
            return item;
        };
    }
    /**
     * It initializes console property as an instance of the internal version of Console() class (PR #748)
     * or as the global node console if the `POWERTOOLS_DEV' env variable is set and has truthy value.
     *
     * @private
     * @returns {void}
     */
    setConsole() {
        if (!this.getEnvVarsService().isDevMode()) {
            this.console = new console_1.Console({ stdout: process.stdout, stderr: process.stderr });
        }
        else {
            this.console = console;
        }
    }
    /**
     * Sets the Logger's customer config service instance, which will be used
     * to fetch environment variables.
     *
     * @private
     * @param {ConfigServiceInterface} customConfigService
     * @returns {void}
     */
    setCustomConfigService(customConfigService) {
        this.customConfigService = customConfigService ? customConfigService : undefined;
    }
    /**
     * Sets the Logger's custom config service instance, which will be used
     * to fetch environment variables.
     *
     * @private
     * @param {ConfigServiceInterface} customConfigService
     * @returns {void}
     */
    setEnvVarsService() {
        this.envVarsService = new config_1.EnvironmentVariablesService();
    }
    /**
     * If the log event feature is enabled via env variable, it sets a property that tracks whether
     * the event passed to the Lambda function handler should be logged or not.
     *
     * @private
     * @returns {void}
     */
    setLogEvent() {
        if (this.getEnvVarsService().getLogEvent()) {
            this.logEvent = true;
        }
    }
    /**
     * It sets the log formatter instance, in charge of giving a custom format
     * to the structured logs
     *
     * @private
     * @param {LogFormatterInterface} logFormatter
     * @returns {void}
     */
    setLogFormatter(logFormatter) {
        this.logFormatter = logFormatter || new formatter_1.PowertoolLogFormatter();
    }
    /**
     * If the `POWERTOOLS_DEV' env variable is set,
     * it adds JSON indentation for pretty printing logs.
     *
     * @private
     * @returns {void}
     */
    setLogIndentation() {
        if (this.getEnvVarsService().isDevMode()) {
            this.logIndentation = 4 /* PRETTY */;
        }
    }
    /**
     * It sets the Logger's instance log level.
     *
     * @private
     * @param {LogLevel} logLevel
     * @returns {void}
     */
    setLogLevel(logLevel) {
        var _a;
        if (this.isValidLogLevel(logLevel)) {
            this.logLevel = logLevel.toUpperCase();
            return;
        }
        const customConfigValue = (_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getLogLevel();
        if (this.isValidLogLevel(customConfigValue)) {
            this.logLevel = customConfigValue.toUpperCase();
            return;
        }
        const envVarsValue = this.getEnvVarsService().getLogLevel();
        if (this.isValidLogLevel(envVarsValue)) {
            this.logLevel = envVarsValue.toUpperCase();
            return;
        }
        this.logLevel = Logger.defaultLogLevel;
    }
    /**
     * If the sample rate feature is enabled, it sets a property that tracks whether this Lambda function invocation
     * will print logs or not.
     *
     * @private
     * @returns {void}
     */
    setLogsSampled() {
        const sampleRateValue = this.getSampleRateValue();
        // TODO: revisit Math.random() as it's not a real randomization
        this.logsSampled = sampleRateValue !== undefined && (sampleRateValue === 1 || Math.random() < sampleRateValue);
    }
    /**
     * It configures the Logger instance settings that will affect the Logger's behaviour
     * and the content of all logs.
     *
     * @private
     * @param {ConstructorOptions} options
     * @returns {Logger}
     */
    setOptions(options) {
        const { logLevel, serviceName, sampleRateValue, logFormatter, customConfigService, persistentLogAttributes, environment, } = options;
        this.setEnvVarsService();
        // order is important, it uses EnvVarsService()
        this.setConsole();
        this.setCustomConfigService(customConfigService);
        this.setLogLevel(logLevel);
        this.setSampleRateValue(sampleRateValue);
        this.setLogsSampled();
        this.setLogFormatter(logFormatter);
        this.setPowertoolLogData(serviceName, environment);
        this.setLogEvent();
        this.setLogIndentation();
        this.addPersistentLogAttributes(persistentLogAttributes);
        return this;
    }
    /**
     * It adds important data to the Logger instance that will affect the content of all logs.
     *
     * @param {string} serviceName
     * @param {Environment} environment
     * @param {LogAttributes} persistentLogAttributes
     * @private
     * @returns {void}
     */
    setPowertoolLogData(serviceName, environment, persistentLogAttributes = {}) {
        var _a, _b;
        this.addToPowertoolLogData({
            awsRegion: this.getEnvVarsService().getAwsRegion(),
            environment: environment ||
                ((_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getCurrentEnvironment()) ||
                this.getEnvVarsService().getCurrentEnvironment(),
            sampleRateValue: this.getSampleRateValue(),
            serviceName: serviceName || ((_b = this.getCustomConfigService()) === null || _b === void 0 ? void 0 : _b.getServiceName()) || this.getEnvVarsService().getServiceName() || this.getDefaultServiceName(),
        }, persistentLogAttributes);
    }
    /**
     * It checks whether the current log item should/can be printed.
     *
     * @param {string} serviceName
     * @param {Environment} environment
     * @param {LogAttributes} persistentLogAttributes
     * @private
     * @returns {boolean}
     */
    shouldPrint(logLevel) {
        if (this.logLevelThresholds[logLevel] >= this.logLevelThresholds[this.getLogLevel()]) {
            return true;
        }
        return this.getLogsSampled();
    }
}
exports.Logger = Logger;
Logger.defaultLogLevel = 'INFO';
//# sourceMappingURL=data:application/json;base64,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