"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricUnits = exports.Metrics = void 0;
const commons_1 = require("@aws-lambda-powertools/commons");
const config_1 = require("./config");
const types_1 = require("./types");
Object.defineProperty(exports, "MetricUnits", { enumerable: true, get: function () { return types_1.MetricUnits; } });
const MAX_METRICS_SIZE = 100;
const MAX_DIMENSION_COUNT = 29;
const DEFAULT_NAMESPACE = 'default_namespace';
/**
 * ## Intro
 * Metrics creates custom metrics asynchronously by logging metrics to standard output following Amazon CloudWatch Embedded Metric Format (EMF).
 *
 * These metrics can be visualized through Amazon CloudWatch Console.
 *
 * ## Key features
 *   * Aggregate up to 100 metrics using a single CloudWatch EMF object (large JSON blob)
 *   * Validate against common metric definitions mistakes (metric unit, values, max dimensions, max metrics, etc)
 *   * Metrics are created asynchronously by CloudWatch service, no custom stacks needed
 *   * Context manager to create a one off metric with a different dimension
 *
 * ## Usage
 *
 * ### Functions usage with middleware
 *
 * Using this middleware on your handler function will automatically flush metrics after the function returns or throws an error.
 * Additionally, you can configure the middleware to easily:
 * * ensure that at least one metric is emitted before you flush them
 * * capture a `ColdStart` a metric
 * * set default dimensions for all your metrics
 *
 * @example
 * ```typescript
 * import { Metrics, logMetrics } from '@aws-lambda-powertools/metrics';
 * import middy from '@middy/core';
 *
 * const metrics = new Metrics({ namespace: 'serverlessAirline', serviceName: 'orders' });
 *
 * const lambdaHandler = async (_event: any, _context: any) => {
 *   ...
 * };
 *
 * export const handler = middy(lambdaHandler).use(logMetrics(metrics));
 * ```
 *
 * ### Object oriented way with decorator
 *
 * If you are used to TypeScript Class usage to encapsulate your Lambda handler you can leverage the [@metrics.logMetrics()](./_aws_lambda_powertools_metrics.Metrics.html#logMetrics) decorator to automatically:
 *   * capture a `ColdStart` metric
 *   * flush buffered metrics
 *   * throw on empty metrics
 *
 * @example
 *
 * ```typescript
 * import { Metrics, MetricUnits } from '@aws-lambda-powertools/metrics';
 * import { LambdaInterface } from '@aws-lambda-powertools/commons';
 *
 * const metrics = new Metrics({ namespace: 'serverlessAirline', serviceName: 'orders' });
 *
 * class Lambda implements LambdaInterface {
 *
 *   // FYI: Decorator might not render properly in VSCode mouse over due to https://github.com/microsoft/TypeScript/issues/47679 and might show as *@metrics* instead of `@metrics.logMetrics`
 *
 *   @metrics.logMetrics({ captureColdStartMetric: true, throwOnEmptyMetrics: true })
 *   public handler(_event: any, _context: any): Promise<void> {
 *     // ...
 *     metrics.addMetric('test-metric', MetricUnits.Count, 10);
 *     // ...
 *   }
 * }
 *
 * const handlerClass = new Lambda();
 * export const handler = handlerClass.handler.bind(handlerClass);
 * ```
 *
 * ### Standard function
 *
 * If you are used to classic JavaScript functions, you can leverage the different methods provided to create and publish metrics.
 *
 * @example
 *
 * ```typescript
 * import { Metrics, MetricUnits } from '@aws-lambda-powertools/metrics';
 *
 * const metrics = new Metrics({ namespace: 'serverlessAirline', serviceName: 'orders' });
 *
 * export const handler = async (_event: any, _context: any): Promise<void> => {
 *   metrics.captureColdStartMetric();
 *   metrics.addMetric('test-metric', MetricUnits.Count, 10);
 *   metrics.publishStoredMetrics();
 * };
 * ```
 */
class Metrics extends commons_1.Utility {
    constructor(options = {}) {
        super();
        this.defaultDimensions = {};
        this.dimensions = {};
        this.isSingleMetric = false;
        this.metadata = {};
        this.shouldThrowOnEmptyMetrics = false;
        this.storedMetrics = {};
        this.dimensions = {};
        this.setOptions(options);
    }
    /**
     * Add a dimension to the metrics.
     * A dimension is a key-value pair that is used to group metrics.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#Dimension for more details.
     * @param name
     * @param value
     */
    addDimension(name, value) {
        if (MAX_DIMENSION_COUNT <= this.getCurrentDimensionsCount()) {
            throw new RangeError(`The number of metric dimensions must be lower than ${MAX_DIMENSION_COUNT}`);
        }
        this.dimensions[name] = value;
    }
    /**
     * Add multiple dimensions to the metrics.
     * @param dimensions
     */
    addDimensions(dimensions) {
        const newDimensions = { ...this.dimensions };
        Object.keys(dimensions).forEach((dimensionName) => {
            newDimensions[dimensionName] = dimensions[dimensionName];
        });
        if (Object.keys(newDimensions).length > MAX_DIMENSION_COUNT) {
            throw new RangeError(`Unable to add ${Object.keys(dimensions).length} dimensions: the number of metric dimensions must be lower than ${MAX_DIMENSION_COUNT}`);
        }
        this.dimensions = newDimensions;
    }
    /**
     * A high-cardinality data part of your Metrics log. This is useful when you want to search highly contextual information along with your metrics in your logs.
     * @param key
     * @param value
     */
    addMetadata(key, value) {
        this.metadata[key] = value;
    }
    /**
     * Add a metric to the metrics buffer.
     * @param name
     * @param unit
     * @param value
     */
    addMetric(name, unit, value) {
        this.storeMetric(name, unit, value);
        if (this.isSingleMetric)
            this.publishStoredMetrics();
    }
    /**
     * Create a singleMetric to capture cold start.
     * If it's a cold start invocation, this feature will:
     *   * Create a separate EMF blob solely containing a metric named ColdStart
     *   * Add function_name and service dimensions
     *
     * This has the advantage of keeping cold start metric separate from your application metrics, where you might have unrelated dimensions.
     *
     * @example
     *
     * ```typescript
     * import { Metrics } from '@aws-lambda-powertools/metrics';
     *
     * const metrics = new Metrics({ namespace: 'serverlessAirline', serviceName: 'orders' });
     *
     * export const handler = async (event: any, _context: any): Promise<void> => {
     *     metrics.captureColdStartMetric();
     * };
     * ```
     */
    captureColdStartMetric() {
        if (!this.isColdStart())
            return;
        const singleMetric = this.singleMetric();
        if (this.defaultDimensions.service) {
            singleMetric.setDefaultDimensions({ service: this.defaultDimensions.service });
        }
        if (this.functionName != null) {
            singleMetric.addDimension('function_name', this.functionName);
        }
        singleMetric.addMetric('ColdStart', types_1.MetricUnits.Count, 1);
    }
    clearDefaultDimensions() {
        this.defaultDimensions = {};
    }
    clearDimensions() {
        this.dimensions = {};
    }
    clearMetadata() {
        this.metadata = {};
    }
    clearMetrics() {
        this.storedMetrics = {};
    }
    /**
     * A decorator automating coldstart capture, throw on empty metrics and publishing metrics on handler exit.
     *
     * @example
     *
     * ```typescript
     * import { Metrics } from '@aws-lambda-powertools/metrics';
     * import { LambdaInterface } from '@aws-lambda-powertools/commons';
     *
     * const metrics = new Metrics({ namespace: 'serverlessAirline', serviceName: 'orders' });
     *
     * class Lambda implements LambdaInterface {
     *
     *   @metrics.logMetrics({ captureColdStartMetric: true })
     *   public handler(_event: any, _context: any): Promise<void> {
     *    // ...
     *   }
     * }
     *
     * const handlerClass = new Lambda();
     * export const handler = handlerClass.handler.bind(handlerClass);
     * ```
     *
     * @decorator Class
     */
    logMetrics(options = {}) {
        const { throwOnEmptyMetrics, defaultDimensions, captureColdStartMetric } = options;
        if (throwOnEmptyMetrics) {
            this.throwOnEmptyMetrics();
        }
        if (defaultDimensions !== undefined) {
            this.setDefaultDimensions(defaultDimensions);
        }
        return (_target, _propertyKey, descriptor) => {
            /**
             * The descriptor.value is the method this decorator decorates, it cannot be undefined.
             */
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            const originalMethod = descriptor.value;
            // eslint-disable-next-line @typescript-eslint/no-this-alias
            const metricsRef = this;
            // Use a function() {} instead of an () => {} arrow function so that we can
            // access `myClass` as `this` in a decorated `myClass.myMethod()`.
            descriptor.value = (async function (event, context, callback) {
                metricsRef.functionName = context.functionName;
                if (captureColdStartMetric)
                    metricsRef.captureColdStartMetric();
                let result;
                try {
                    result = await originalMethod.apply(this, [event, context, callback]);
                }
                catch (error) {
                    throw error;
                }
                finally {
                    metricsRef.publishStoredMetrics();
                }
                return result;
            });
            return descriptor;
        };
    }
    /**
     * Synchronous function to actually publish your metrics. (Not needed if using logMetrics decorator).
     * It will create a new EMF blob and log it to standard output to be then ingested by Cloudwatch logs and processed automatically for metrics creation.
     *
     * @example
     *
     * ```typescript
     * import { Metrics, MetricUnits } from '@aws-lambda-powertools/metrics';
     *
     * const metrics = new Metrics({ namespace: 'serverlessAirline', serviceName: 'orders' }); // Sets metric namespace, and service as a metric dimension
     *
     * export const handler = async (_event: any, _context: any): Promise<void> => {
     *   metrics.addMetric('test-metric', MetricUnits.Count, 10);
     *   metrics.publishStoredMetrics();
     * };
     * ```
     */
    publishStoredMetrics() {
        const target = this.serializeMetrics();
        console.log(JSON.stringify(target));
        this.clearMetrics();
        this.clearDimensions();
        this.clearMetadata();
    }
    /**
     * Function to create the right object compliant with Cloudwatch EMF (Event Metric Format).
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Embedded_Metric_Format_Specification.html for more details
     * @returns {string}
     */
    serializeMetrics() {
        const metricDefinitions = Object.values(this.storedMetrics).map((metricDefinition) => ({
            Name: metricDefinition.name,
            Unit: metricDefinition.unit,
        }));
        if (metricDefinitions.length === 0 && this.shouldThrowOnEmptyMetrics) {
            throw new RangeError('The number of metrics recorded must be higher than zero');
        }
        if (!this.namespace)
            console.warn('Namespace should be defined, default used');
        const metricValues = Object.values(this.storedMetrics).reduce((result, { name, value }) => {
            result[name] = value;
            return result;
        }, {});
        const dimensionNames = [...Object.keys(this.defaultDimensions), ...Object.keys(this.dimensions)];
        return {
            _aws: {
                Timestamp: new Date().getTime(),
                CloudWatchMetrics: [
                    {
                        Namespace: this.namespace || DEFAULT_NAMESPACE,
                        Dimensions: [dimensionNames],
                        Metrics: metricDefinitions,
                    },
                ],
            },
            ...this.defaultDimensions,
            ...this.dimensions,
            ...metricValues,
            ...this.metadata,
        };
    }
    setDefaultDimensions(dimensions) {
        const targetDimensions = {
            ...this.defaultDimensions,
            ...dimensions,
        };
        if (MAX_DIMENSION_COUNT <= Object.keys(targetDimensions).length) {
            throw new Error('Max dimension count hit');
        }
        this.defaultDimensions = targetDimensions;
    }
    setFunctionName(value) {
        this.functionName = value;
    }
    /**
     * CloudWatch EMF uses the same dimensions across all your metrics. Use singleMetric if you have a metric that should have different dimensions.
     *
     * You don't need to call publishStoredMetrics() after calling addMetric for a singleMetrics, they will be flushed directly.
     *
     * @example
     *
     * ```typescript
     * const singleMetric = metrics.singleMetric();
     * singleMetric.addDimension('InnerDimension', 'true');
     * singleMetric.addMetric('single-metric', MetricUnits.Percent, 50);
     * ```
     *
     * @returns the Metrics
     */
    singleMetric() {
        return new Metrics({
            namespace: this.namespace,
            serviceName: this.dimensions.service,
            defaultDimensions: this.defaultDimensions,
            singleMetric: true,
        });
    }
    /**
     * Throw an Error if the metrics buffer is empty.
     *
     * @example
     *
     * ```typescript
     * import { Metrics } from '@aws-lambda-powertools/metrics';
     *
     * const metrics = new Metrics({ namespace: 'serverlessAirline', serviceName:'orders' });
     *
     * export const handler = async (_event: any, _context: any): Promise<void> => {
     *     metrics.throwOnEmptyMetrics();
     *     metrics.publishStoredMetrics(); // will throw since no metrics added.
     * };
     * ```
     */
    throwOnEmptyMetrics() {
        this.shouldThrowOnEmptyMetrics = true;
    }
    getCurrentDimensionsCount() {
        return Object.keys(this.dimensions).length + Object.keys(this.defaultDimensions).length;
    }
    getCustomConfigService() {
        return this.customConfigService;
    }
    getEnvVarsService() {
        return this.envVarsService;
    }
    isNewMetric(name, unit) {
        if (this.storedMetrics[name]) {
            // Inconsistent units indicates a bug or typos and we want to flag this to users early
            if (this.storedMetrics[name].unit !== unit) {
                const currentUnit = this.storedMetrics[name].unit;
                throw new Error(`Metric "${name}" has already been added with unit "${currentUnit}", but we received unit "${unit}". Did you mean to use metric unit "${currentUnit}"?`);
            }
            return false;
        }
        else {
            return true;
        }
    }
    setCustomConfigService(customConfigService) {
        this.customConfigService = customConfigService ? customConfigService : undefined;
    }
    setEnvVarsService() {
        this.envVarsService = new config_1.EnvironmentVariablesService();
    }
    setNamespace(namespace) {
        var _a;
        this.namespace = (namespace ||
            ((_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getNamespace()) ||
            this.getEnvVarsService().getNamespace());
    }
    setOptions(options) {
        const { customConfigService, namespace, serviceName, singleMetric, defaultDimensions } = options;
        this.setEnvVarsService();
        this.setCustomConfigService(customConfigService);
        this.setNamespace(namespace);
        this.setService(serviceName);
        this.setDefaultDimensions(defaultDimensions);
        this.isSingleMetric = singleMetric || false;
        return this;
    }
    setService(service) {
        var _a;
        const targetService = (service ||
            ((_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getServiceName()) ||
            this.getEnvVarsService().getServiceName()) || this.getDefaultServiceName();
        if (targetService.length > 0) {
            this.setDefaultDimensions({ service: targetService });
        }
    }
    storeMetric(name, unit, value) {
        if (Object.keys(this.storedMetrics).length >= MAX_METRICS_SIZE) {
            this.publishStoredMetrics();
        }
        if (this.isNewMetric(name, unit)) {
            this.storedMetrics[name] = {
                unit,
                value,
                name,
            };
        }
        else {
            const storedMetric = this.storedMetrics[name];
            if (!Array.isArray(storedMetric.value)) {
                storedMetric.value = [storedMetric.value];
            }
            storedMetric.value.push(value);
        }
    }
}
exports.Metrics = Metrics;
//# sourceMappingURL=data:application/json;base64,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