"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareSecuritySchemes = exports.getAllAuthorizers = exports.applyMethodAuthorizer = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const authorizers_1 = require("../authorizers");
const predicates_1 = require("../authorizers/predicates");
const utils_1 = require("./utils");
// Regex to match against a single header identity source
const SINGLE_HEADER_IDENTITY_SOURCE_REGEX = /^method.request.header.(?<header>[^\.\s,]+)$/;
/**
 * Create the OpenAPI definition with api gateway extensions for the given authorizer
 * @param authorizer the authorizer used for the method
 */
exports.applyMethodAuthorizer = (authorizer) => {
    if (predicates_1.isCustomAuthorizer(authorizer) || predicates_1.isCognitoAuthorizer(authorizer)) {
        return {
            security: [
                {
                    [authorizer.authorizerId]: predicates_1.isCognitoAuthorizer(authorizer)
                        ? authorizer.authorizationScopes
                        : [],
                },
            ],
        };
    }
    // IAM and NONE are specified via x-amazon-apigateway-auth
    return {
        "x-amazon-apigateway-auth": {
            type: authorizer.authorizationType,
        },
    };
};
/**
 * Create an OpenAPI security scheme definition for a cognito authorizer
 * @param authorizer cognito authorizer
 */
const cognitoSecurityScheme = (authorizer) => ({
    type: "apiKey",
    name: "Authorization",
    in: "header",
    "x-amazon-apigateway-authtype": authorizer.authorizationType,
    "x-amazon-apigateway-authorizer": {
        type: aws_apigateway_1.AuthorizationType.COGNITO,
        providerARNs: authorizer.userPools.map((pool) => pool.userPoolArn),
    },
});
/**
 * Create an OpenAPI security scheme definition for a custom authorizer
 * @param scope the scope in which the api construct is defined
 * @param authorizer custom authorizer
 */
const customSecurityScheme = (scope, authorizer) => {
    const singleHeaderMatch = authorizer.identitySource.match(SINGLE_HEADER_IDENTITY_SOURCE_REGEX);
    const singleHeader = singleHeaderMatch
        ? singleHeaderMatch.groups.header
        : undefined;
    // Token type must use a single header
    if (!singleHeader && authorizer.type === authorizers_1.CustomAuthorizerType.TOKEN) {
        throw new Error(`identitySource must be a single header for a ${authorizers_1.CustomAuthorizerType.TOKEN} authorizer, eg method.request.header.Authorization`);
    }
    return {
        type: "apiKey",
        in: "header",
        // When the identity source is not a single header, the value must be "Unused"
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-authorizer.html
        name: singleHeader ? singleHeader : "Unused",
        "x-amazon-apigateway-authtype": authorizer.authorizationType,
        "x-amazon-apigateway-authorizer": {
            type: authorizer.type,
            authorizerUri: utils_1.functionInvocationUri(scope, authorizer.function),
            authorizerResultTtlInSeconds: authorizer.authorizerResultTtlInSeconds,
            identitySource: authorizer.identitySource,
        },
    };
};
/**
 * Return a list of all authorizers used in the api
 */
exports.getAllAuthorizers = (options) => [
    ...(options.defaultAuthorizer ? [options.defaultAuthorizer] : []),
    ...Object.values(options.integrations).flatMap(({ authorizer }) => authorizer ? [authorizer] : []),
];
/**
 * Generate the security schemes section of an OpenAPI specification
 */
exports.prepareSecuritySchemes = (scope, options) => {
    // All the defined authorizers
    const allAuthorizers = exports.getAllAuthorizers(options);
    // Cognito and custom authorizers must be declared in security schemes
    return {
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => predicates_1.isCognitoAuthorizer(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            cognitoSecurityScheme(authorizer),
        ])),
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => predicates_1.isCustomAuthorizer(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            customSecurityScheme(scope, authorizer),
        ])),
    };
};
//# sourceMappingURL=data:application/json;base64,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