"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const apigateway = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const duration_1 = require("@aws-cdk/core/lib/duration");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const core_2 = require("@aws-solutions-constructs/core");
// Setup
const app = new core_1.App();
const stack = new core_1.Stack(app, core_2.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway-lambda';
const lambdaProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: lambda.Runtime.NODEJS_10_X,
    handler: 'index.handler'
};
// Some Caching for static content
const someCachePolicy = new cloudfront.CachePolicy(stack, 'SomeCachePolicy', {
    cachePolicyName: 'SomeCachePolicy',
    defaultTtl: duration_1.Duration.hours(8),
    minTtl: duration_1.Duration.hours(5),
    maxTtl: duration_1.Duration.hours(10),
});
// Disable Caching for dynamic content
const noCachePolicy = new cloudfront.CachePolicy(stack, 'NoCachePolicy', {
    cachePolicyName: 'NoCachePolicy',
    defaultTtl: duration_1.Duration.minutes(0),
    minTtl: duration_1.Duration.minutes(0),
    maxTtl: duration_1.Duration.minutes(0),
});
const construct = new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cf-api-lambda-override', {
    lambdaFunctionProps: lambdaProps,
    apiGatewayProps: {
        proxy: false,
        defaultMethodOptions: {
            authorizationType: apigateway.AuthorizationType.NONE,
        },
    },
    cloudFrontDistributionProps: {
        defaultBehavior: {
            cachePolicy: someCachePolicy
        }
    }
});
const apiEndPoint = construct.apiGateway;
const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
const staticResource = construct.apiGateway.root.addResource('static');
const staticMethod = staticResource.addMethod('GET', new apigateway.HttpIntegration('http://amazon.com'));
const dynamicResource = construct.apiGateway.root.addResource('dynamic');
const dynamicMethod = dynamicResource.addMethod('GET');
// Add behavior
construct.cloudFrontWebDistribution.addBehavior('/dynamic', new origins.HttpOrigin(apiEndPointDomainName, {
    originPath: `/${apiEndPoint.deploymentStage.stageName}/dynamic`
}), {
    cachePolicy: noCachePolicy
});
// Suppress CFN_NAG warnings
suppressWarnings(staticMethod);
suppressWarnings(dynamicMethod);
function suppressWarnings(method) {
    const child = method.node.findChild('Resource');
    child.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W59',
                    reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                }]
        }
    };
}
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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