"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stageDependencies = exports.bundle = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
function bundle(options) {
    const { entry, runtime, outputPathSuffix } = options;
    const stagedir = cdk.FileSystem.mkdtemp('python-bundling-');
    const hasDeps = stageDependencies(entry, stagedir);
    const depsCommand = chain([
        hasDeps ? `rsync -r ${exports.BUNDLER_DEPENDENCIES_CACHE}/. ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}` : '',
        `rsync -r . ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}`,
    ]);
    // Determine which dockerfile to use. When dependencies are present, we use a
    // Dockerfile that can create a cacheable layer. We can't use this Dockerfile
    // if there aren't dependencies or the Dockerfile will complain about missing
    // sources.
    const dockerfile = hasDeps
        ? 'Dockerfile.dependencies'
        : 'Dockerfile';
    // copy Dockerfile to workdir
    fs.copyFileSync(path.join(__dirname, dockerfile), path.join(stagedir, dockerfile));
    const image = cdk.DockerImage.fromBuild(stagedir, {
        buildArgs: {
            IMAGE: runtime.bundlingImage.image,
        },
        file: dockerfile,
    });
    return lambda.Code.fromAsset(entry, {
        assetHashType: options.assetHashType,
        assetHash: options.assetHash,
        exclude: exports.DEPENDENCY_EXCLUDES,
        bundling: {
            image,
            command: ['bash', '-c', depsCommand],
        },
    });
}
exports.bundle = bundle;
/**
 * Checks to see if the `entry` directory contains a type of dependency that
 * we know how to install.
 */
function stageDependencies(entry, stagedir) {
    const prefixes = [
        'Pipfile',
        'pyproject',
        'poetry',
        'requirements.txt',
    ];
    let found = false;
    for (const file of fs.readdirSync(entry)) {
        for (const prefix of prefixes) {
            if (file.startsWith(prefix)) {
                fs.copyFileSync(path.join(entry, file), path.join(stagedir, file));
                found = true;
            }
        }
    }
    return found;
}
exports.stageDependencies = stageDependencies;
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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