"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayWebAcl = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_wafv2_1 = require("aws-cdk-lib/aws-wafv2");
const constructs_1 = require("constructs");
/**
 * Associate an AWS WAF v2 Web ACL with the given api
 */
class OpenApiGatewayWebAcl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const aclName = `${pdk_nag_1.PDKNag.getStackPrefix(aws_cdk_lib_1.Stack.of(this))
            .split("/")
            .join("-")}-${id}-WebAcl`;
        const ipSetName = `${aclName}-IPSet`;
        // Create the IP Set if requested
        this.ipSet = props.cidrAllowList
            ? new aws_wafv2_1.CfnIPSet(this, "IPSet", {
                name: ipSetName,
                addresses: props.cidrAllowList.cidrRanges,
                ipAddressVersion: props.cidrAllowList.cidrType,
                scope: "REGIONAL",
            })
            : undefined;
        const managedRules = props.managedRules ?? [
            { vendor: "AWS", name: "AWSManagedRulesCommonRuleSet" },
        ];
        const rules = [
            // Add a rule for the IP Set if specified
            ...(this.ipSet
                ? [
                    {
                        name: ipSetName,
                        priority: 1,
                        visibilityConfig: {
                            metricName: ipSetName,
                            cloudWatchMetricsEnabled: true,
                            sampledRequestsEnabled: true,
                        },
                        action: {
                            block: {},
                        },
                        statement: {
                            notStatement: {
                                statement: {
                                    ipSetReferenceStatement: {
                                        arn: this.ipSet.attrArn,
                                    },
                                },
                            },
                        },
                    },
                ]
                : []),
            // Add the managed rules
            ...managedRules.map(({ vendor, name }, i) => ({
                name: `${vendor}-${name}`,
                priority: i + 2,
                statement: { managedRuleGroupStatement: { vendorName: vendor, name } },
                overrideAction: { none: {} },
                visibilityConfig: {
                    metricName: `${aclName}-${vendor}-${name}`,
                    cloudWatchMetricsEnabled: true,
                    sampledRequestsEnabled: true,
                },
            })),
        ];
        this.webAcl = new aws_wafv2_1.CfnWebACL(this, "WebACL", {
            name: aclName,
            defaultAction: {
                // Allow by default, and use rules to deny unwanted requests
                allow: {},
            },
            scope: "REGIONAL",
            visibilityConfig: {
                cloudWatchMetricsEnabled: true,
                sampledRequestsEnabled: true,
                metricName: aclName,
            },
            rules,
        });
        this.webAclAssociation = new aws_wafv2_1.CfnWebACLAssociation(this, "WebACLAssociation", {
            resourceArn: props.apiDeploymentStageArn,
            webAclArn: this.webAcl.attrArn,
        });
    }
}
exports.OpenApiGatewayWebAcl = OpenApiGatewayWebAcl;
//# sourceMappingURL=data:application/json;base64,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