"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.invokeCustomDocsGenerator = exports.invokeOpenApiGenerator = exports.NonClientGeneratorDirectory = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const util_1 = require("projen/lib/util");
/**
 * Enum for generator directories for non-client generators
 */
var NonClientGeneratorDirectory;
(function (NonClientGeneratorDirectory) {
    NonClientGeneratorDirectory["DOCS"] = "docs";
})(NonClientGeneratorDirectory = exports.NonClientGeneratorDirectory || (exports.NonClientGeneratorDirectory = {}));
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
/**
 * Clean up any files that have already been generated by the openapi generator
 * @param generatedOutputPath output path for the generator
 */
const cleanPreviouslyGeneratedFiles = (generatedOutputPath) => {
    // OpenAPI generator writes a manifest called FILES which lists the files it generated.
    const openApiGeneratedFilesManifestPath = path.join(generatedOutputPath, ".openapi-generator", "FILES");
    // If the manifest exists, delete the files it lists
    if (fs.existsSync(openApiGeneratedFilesManifestPath)) {
        const previouslyGeneratedFiles = new Set(fs
            .readFileSync(openApiGeneratedFilesManifestPath, { encoding: "utf-8" })
            .split("\n")
            .filter((x) => x));
        previouslyGeneratedFiles.forEach((previouslyGeneratedFile) => {
            fs.unlinkSync(path.join(generatedOutputPath, previouslyGeneratedFile));
        });
    }
};
/**
 * Generate client code or docs by invoking the root generate script
 */
exports.invokeOpenApiGenerator = (options) => {
    // Prior to generating, clean up any previously generated files so that we don't have stale generated code from
    // previous executions.
    cleanPreviouslyGeneratedFiles(options.outputPath);
    const srcDir = options.srcDir ?? "src";
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    util_1.exec(`./generate --generator ${options.generator} --spec-path ${options.specPath} --output-path ${options.outputPath} --generator-dir ${options.generatorDirectory} --src-dir ${srcDir}${additionalProperties}`, {
        cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "generators"),
    });
};
/**
 * Invoke a custom documentation generator script
 */
exports.invokeCustomDocsGenerator = (options) => {
    util_1.exec(`./${options.generator}${options.args ? ` ${options.args}` : ""}`, {
        cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "custom", "docs"),
    });
};
//# sourceMappingURL=data:application/json;base64,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