"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPythonSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample python source code
 */
exports.getPythonSampleSource = (options) => ({
    "__init__.py": "#",
    // This file provides a type-safe interface to the exported OpenApiGatewayRestApi construct
    "api.py": `from dataclasses import fields
from ${options.openApiGatewayPackageName} import OpenApiGatewayRestApi, OpenApiIntegration
from ${options.pythonClientPackageName}.apis.tags.default_api_operation_config import OperationLookup, OperationConfig
from ${options.moduleName}.spec_utils import SPEC, SPEC_PATH

class Api(OpenApiGatewayRestApi):
    """
    Type-safe construct for the API Gateway resources defined by the spec.
    You will likely not need to modify this file, and can instead extend it and define your integrations.
    """
    def __init__(self, scope, id, integrations: OperationConfig[OpenApiIntegration], **kwargs):
        super().__init__(scope, id,
            **kwargs,
            integrations={ field.name: getattr(integrations, field.name) for field in fields(integrations) },
            spec=SPEC,
            spec_path=SPEC_PATH,
            operation_lookup=OperationLookup,
        )
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample_api.py": `from ${options.openApiGatewayPackageName} import Authorizers, Integrations, OpenApiIntegration
from ${options.pythonClientPackageName}.apis.tags.default_api_operation_config import OperationConfig
from aws_cdk.aws_lambda import LayerVersion, Code, Function, Runtime
from .api import Api
from constructs import Construct
from ${options.moduleName}.spec_utils import get_generated_client_layer_directory
from pathlib import Path
from os import path

class SampleApi(Construct):
  """
  An example of how to add integrations to your api
  """
  def __init__(self, scope, id):
      super().__init__(scope, id)

      # Layer which contains the generated client.
      self.generated_client_layer = LayerVersion(self, 'GeneratedClientLayer',
          code=Code.from_asset(get_generated_client_layer_directory())
      )

      self.api = Api(self, 'Api',
          default_authorizer=Authorizers.iam(),
          integrations=OperationConfig(
              say_hello=OpenApiIntegration(
                  integration=Integrations.lambda_(Function(self, 'SayHello',
                      runtime=Runtime.PYTHON_3_9,
                      code=Code.from_asset(path.join(str(Path(__file__).parent.absolute()), 'handlers')),
                      handler="say_hello_handler_sample.handler",
                      layers=[self.generated_client_layer],
                  )),
              ),
          ),
      )
`,
            // Generate an example lambda handler
            "handlers/say_hello_handler_sample.py": `from ${options.pythonClientPackageName}.apis.tags.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse, SayHelloOperationResponses
from ${options.pythonClientPackageName}.model.say_hello_response_content import SayHelloResponseContent

@say_hello_handler
def handler(input: SayHelloRequest, **kwargs) -> SayHelloOperationResponses:
    """
    An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs
    """
    return ApiResponse(
        status_code=200,
        body=SayHelloResponseContent(message="Hello {}!".format(input.request_parameters["name"])),
        headers={}
    )
`,
        }
        : {}),
});
//# sourceMappingURL=data:application/json;base64,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