"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateClientProjects = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const languages_1 = require("../languages");
const generated_java_client_project_1 = require("./generated-java-client-project");
const generated_python_client_project_1 = require("./generated-python-client-project");
const generated_typescript_client_project_1 = require("./generated-typescript-client-project");
/**
 * Returns a generated client project for the given language
 */
const generateClientProject = (language, options) => {
    switch (language) {
        case languages_1.ClientLanguage.TYPESCRIPT:
            return new generated_typescript_client_project_1.GeneratedTypescriptClientProject({
                parent: options.parent,
                // Ensure kebab-case for typescript
                name: `${options.parentPackageName}-${languages_1.ClientLanguage.TYPESCRIPT}`.replace(/_/g, "-"),
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.TYPESCRIPT),
                specPath: options.parsedSpecPath,
                ...options.typescriptOptions,
            });
        case languages_1.ClientLanguage.PYTHON:
            // Ensure snake_case for python
            const pythonName = `${options.parentPackageName}_${languages_1.ClientLanguage.PYTHON}`
                .replace(/@/g, "")
                .replace(/[\-/]/g, "_");
            return new generated_python_client_project_1.GeneratedPythonClientProject({
                parent: options.parent,
                rootProjectHasParent: options.rootProjectHasParent,
                name: pythonName,
                moduleName: pythonName,
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.PYTHON),
                specPath: options.parsedSpecPath,
                ...options.pythonOptions,
            });
        case languages_1.ClientLanguage.JAVA:
            // Ensure no dashes/underscores since name is used in package name
            const javaName = `${options.parentPackageName}-${languages_1.ClientLanguage.JAVA}`
                .replace(/@/g, "")
                .replace(/[\-/_]/g, "");
            const artifactId = `${options.parentPackageName}-${languages_1.ClientLanguage.JAVA}`
                .replace(/@/g, "")
                .replace(/[/_]/g, "-");
            return new generated_java_client_project_1.GeneratedJavaClientProject({
                parent: options.parent,
                name: javaName,
                artifactId,
                groupId: "com.generated.api",
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.JAVA),
                specPath: options.parsedSpecPath,
                ...options.javaOptions,
            });
        default:
            throw new Error(`Unknown client language ${language}`);
    }
};
/**
 * Generate API clients in the given languages
 * @param languages the languages to generate clients for
 * @param options options for the projects to be created
 */
exports.generateClientProjects = (languages, options) => Object.fromEntries([...languages].map((language) => [
    language,
    generateClientProject(language, options),
]));
//# sourceMappingURL=data:application/json;base64,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