"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stageDependencies = exports.bundle = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
function bundle(options) {
    const { entry, runtime, architecture, outputPathSuffix } = options;
    const stagedir = cdk.FileSystem.mkdtemp('python-bundling-');
    const hasDeps = stageDependencies(entry, stagedir);
    const depsCommand = chain([
        hasDeps ? `rsync -r ${exports.BUNDLER_DEPENDENCIES_CACHE}/. ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}` : '',
        `rsync -r . ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}`,
    ]);
    // Determine which dockerfile to use. When dependencies are present, we use a
    // Dockerfile that can create a cacheable layer. We can't use this Dockerfile
    // if there aren't dependencies or the Dockerfile will complain about missing
    // sources.
    const dockerfile = hasDeps
        ? 'Dockerfile.dependencies'
        : 'Dockerfile';
    // copy Dockerfile to workdir
    fs.copyFileSync(path.join(__dirname, dockerfile), path.join(stagedir, dockerfile));
    const image = cdk.DockerImage.fromBuild(stagedir, {
        buildArgs: {
            IMAGE: runtime.bundlingImage.image,
        },
        platform: architecture.dockerPlatform,
        file: dockerfile,
    });
    return lambda.Code.fromAsset(entry, {
        assetHashType: options.assetHashType,
        assetHash: options.assetHash,
        exclude: exports.DEPENDENCY_EXCLUDES,
        bundling: {
            image,
            command: ['bash', '-c', depsCommand],
        },
    });
}
exports.bundle = bundle;
/**
 * Checks to see if the `entry` directory contains a type of dependency that
 * we know how to install.
 */
function stageDependencies(entry, stagedir) {
    const prefixes = [
        'Pipfile',
        'pyproject',
        'poetry',
        'requirements.txt',
    ];
    let found = false;
    for (const file of fs.readdirSync(entry)) {
        for (const prefix of prefixes) {
            if (file.startsWith(prefix)) {
                fs.copyFileSync(path.join(entry, file), path.join(stagedir, file));
                found = true;
            }
        }
    }
    return found;
}
exports.stageDependencies = stageDependencies;
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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