"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const api = require("aws-cdk-lib/aws-apigateway");
require("@aws-cdk/assert/jest");
function deployNewFunc(stack) {
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    return new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
}
test('override cognito properties', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolProps = {
        userPoolName: 'test',
        userVerification: {}
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolProps
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPool', {
        AdminCreateUserConfig: {
            AllowAdminCreateUserOnly: true
        },
        EmailVerificationMessage: "The verification code to your new account is {####}",
        EmailVerificationSubject: "Verify your new account",
        SmsVerificationMessage: "The verification code to your new account is {####}",
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test",
        VerificationMessageTemplate: {
            DefaultEmailOption: "CONFIRM_WITH_CODE",
            EmailMessage: "The verification code to your new account is {####}",
            EmailSubject: "Verify your new account",
            SmsMessage: "The verification code to your new account is {####}"
        }
    });
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.userPool !== null);
    expect(construct.userPoolClient !== null);
    expect(construct.apiGateway !== null);
    expect(construct.lambdaFunction !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
    expect(construct.apiGatewayAuthorizer !== null);
});
test('override cognito cognitoUserPoolClientProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolClientProps = {
        authFlows: {
            userSrp: true
        }
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolClientProps
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
        ExplicitAuthFlows: [
            "ALLOW_USER_SRP_AUTH",
            "ALLOW_REFRESH_TOKEN_AUTH"
        ],
    });
});
test('Check for default Cognito Auth Type', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('override Auth Type to COGNITO', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.COGNITO
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Try to override Auth Type to NONE', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                operationName: 'foo-bar'
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps to support CORS', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultCorsPreflightOptions: {
                allowOrigins: api.Cors.ALL_ORIGINS,
                allowMethods: api.Cors.ALL_METHODS
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "NONE"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "NONE"
    });
});
test('Override apiGatewayProps with proxy = false and add POST method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('POST');
    // Super imporant to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "POST",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
test('Override apiGatewayProps with proxy = false and add OPTIONS method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('OPTIONS');
    // Mandatory to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "NONE",
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
//# sourceMappingURL=data:application/json;base64,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