"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateClientProjects = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const generated_java_client_project_1 = require("./generated-java-client-project");
const generated_python_client_project_1 = require("./generated-python-client-project");
const generated_typescript_client_project_1 = require("./generated-typescript-client-project");
const languages_1 = require("../languages");
const logger = log4js_1.getLogger();
/**
 * Returns a generated client project for the given language
 */
const generateClientProject = (languageConfig, options) => {
    switch (languageConfig.clientLanguage) {
        case languages_1.ClientLanguage.TYPESCRIPT: {
            logger.trace("Attempting to generate TYPESCRIPT client project.");
            return new generated_typescript_client_project_1.GeneratedTypescriptClientProject({
                parent: options.parent,
                generateClient: languageConfig.generate,
                // Ensure kebab-case for typescript
                name: `${options.parentPackageName}-${languages_1.ClientLanguage.TYPESCRIPT}`.replace(/_/g, "-"),
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.TYPESCRIPT),
                specPath: options.parsedSpecPath,
                ...options.typescriptOptions,
            });
        }
        case languages_1.ClientLanguage.PYTHON: {
            logger.trace("Attempting to generate PYTHON client project.");
            // Ensure snake_case for python
            const moduleName = `${options.parentPackageName}_${languages_1.ClientLanguage.PYTHON}`
                .replace(/@/g, "")
                .replace(/[\-/]/g, "_");
            return new generated_python_client_project_1.GeneratedPythonClientProject({
                parent: options.parent,
                generateClient: languageConfig.generate,
                // Use dashes in project name since distributable's PKG-INFO always converts _ to -
                // https://stackoverflow.com/questions/36300788/python-package-wheel-pkg-info-name
                name: moduleName.replace(/_/g, "-"),
                moduleName,
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.PYTHON),
                specPath: options.parsedSpecPath,
                ...options.pythonOptions,
            });
        }
        case languages_1.ClientLanguage.JAVA: {
            logger.trace("Attempting to generate JAVA client project.");
            // Ensure no dashes/underscores since name is used in package name
            const javaProjectName = `${options.parentPackageName}-${languages_1.ClientLanguage.JAVA}`
                .replace(/@/g, "")
                .replace(/[\-/_]/g, "");
            const artifactId = `${options.parentPackageName}-${languages_1.ClientLanguage.JAVA}`
                .replace(/@/g, "")
                .replace(/[/_]/g, "-");
            return new generated_java_client_project_1.GeneratedJavaClientProject({
                parent: options.parent,
                generateClient: languageConfig.generate,
                name: javaProjectName,
                artifactId,
                groupId: "com.generated.api",
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.JAVA),
                specPath: options.parsedSpecPath,
                ...options.javaOptions,
            });
        }
        default:
            throw new Error(`Unknown client language ${languageConfig.clientLanguage}`);
    }
};
/**
 * Generate API clients in the given languages
 * @param languages the languages to generate clients for
 * @param options options for the projects to be created
 */
exports.generateClientProjects = (languageConfigs, options) => {
    new projen_1.TextFile(options.parent, path.join(options.generatedCodeDir, "README.md"), {
        lines: [
            "## Generated Clients",
            "",
            "This directory contains generated client code based on your OpenAPI Specification file (spec.yaml).",
            "",
            "Like other `projen` managed files, this directory should be checked in to source control, but should not be edited manually.",
        ],
        readonly: true,
    });
    const generatedClients = {};
    languageConfigs.forEach((languageConfig) => {
        const project = generateClientProject(languageConfig, options);
        if (project != null) {
            generatedClients[languageConfig.clientLanguage] = project;
        }
    });
    return generatedClients;
};
//# sourceMappingURL=data:application/json;base64,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