# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['diffsync']

package_data = \
{'': ['*']}

install_requires = \
['pydantic>=1.7.2,<2.0.0', 'structlog>=20.1.0,<21.0.0']

extras_require = \
{':python_version >= "3.6" and python_version < "3.7"': ['dataclasses>=0.7,<0.8']}

setup_kwargs = {
    'name': 'diffsync',
    'version': '1.2.0',
    'description': 'Library to easily sync/diff/update 2 different data sources',
    'long_description': '# DiffSync\n\nDiffSync is a utility library that can be used to compare and synchronize different datasets.\n\nFor example, it can be used to compare a list of devices from 2 inventory systems and, if required, synchronize them in either direction.\n\n```python\nA = DiffSyncSystemA()\nB = DiffSyncSystemB()\n\nA.load()\nB.load()\n\n# Show the difference between both systems, that is, what would change if we applied changes from System B to System A\ndiff_a_b = A.diff_from(B)\nprint(diff.str())\n\n# Update System A to align with the current status of system B\nA.sync_from(B)\n\n# Update System B to align with the current status of system A\nA.sync_to(B)\n```\n\n# Getting started\n\nTo be able to properly compare different datasets, DiffSync relies on a shared datamodel that both systems must use.\nSpecifically, each system or dataset must provide a `DiffSync` "adapter" subclass, which in turn represents its dataset as instances of one or more `DiffSyncModel` data model classes.\n\nWhen comparing two systems, DiffSync detects the intersection between the two systems (which data models they have in common, and which attributes are shared between each pair of data models) and uses this intersection to compare and/or synchronize the data.\n\n## Define your model with DiffSyncModel\n\n`DiffSyncModel` is based on [Pydantic](https://pydantic-docs.helpmanual.io/) and is using Python typing to define the format of each attribute.\nEach `DiffSyncModel` subclass supports the following class-level attributes:\n- `_modelname` - Defines the type of the model; used to identify common models between different systems (Mandatory)\n- `_identifiers` - List of instance field names used as primary keys for this object (Mandatory)\n- `_shortname` - List of instance field names to use for a shorter name (Optional)\n- `_attributes` - List of non-identifier instance field names for this object; used to identify the fields in common between data models for different systems (Optional)\n- `_children` - Dict of `{<model_name>: <field_name>}` indicating which fields store references to child data model instances. (Optional)\n\n> DiffSyncModel instances must be uniquely identified by their unique id, composed of all fields defined in `_identifiers`. The unique id must be globally meaningful (such as an unique instance name or slug), as it is used to identify object correspondence between differing systems or data sets. It **must not** be a value that is only locally meaningful, such as a database primary key integer value.\n\n> Only fields listed in `_identifiers`, `_attributes`, or `_children` will be potentially included in comparison and synchronization between systems or data sets. Any other fields will be ignored; this allows for a model to additionally contain fields that are only locally relevant (such as database primary key values) and therefore are irrelevant to comparisons.\n\n```python\nfrom typing import List, Optional\nfrom diffsync import DiffSyncModel\n\nclass Site(DiffSyncModel):\n    _modelname = "site"\n    _identifiers = ("name",)\n    _shortname = ()\n    _attributes = ("contact_phone",)\n    _children = {"device": "devices"}\n\n    name: str\n    contact_phone: Optional[str]\n    devices: List = list()\n    database_pk: Optional[int]  # not listed in _identifiers/_attributes/_children as it\'s only locally significant\n```\n\n### Relationship between models\n\nCurrently the relationships between models are very loose by design. Instead of storing an object, it\'s recommended to store the unique id of an object and retrieve it from the store as needed. The `add_child()` API of `DiffSyncModel` provides this behavior as a default.\n\n## Define your system adapter with DiffSync\n\nA `DiffSync` "adapter" subclass must reference each model available at the top of the object by its modelname and must have a `top_level` attribute defined to indicate how the diff and the synchronization should be done. In the example below, `"site"` is the only top level object so the synchronization engine will only check all known `Site` instances and all children of each Site. In this case, as shown in the code above, `Device`s are children of `Site`s, so this is exactly the intended logic.\n\n```python\nfrom diffsync import DiffSync\n\nclass BackendA(DiffSync):\n\n    site = Site\n    device = Device\n\n    top_level = ["site"]\n```\n\nIt\'s up to the implementer to populate the `DiffSync`\'s internal cache with the appropriate data. In the example below we are using the `load()` method to populate the cache but it\'s not mandatory, it could be done differently.\n\n## Store data in a `DiffSync` object\n\nTo add a site to the local cache/store, you need to pass a valid `DiffSyncModel` object to the `add()` function.\n\n```python\nclass BackendA(DiffSync):\n    [...]\n\n    def load(self):\n        # Store an individual object\n        site = self.site(name="nyc")\n        self.add(site)\n\n        # Store an object and define it as a child of another object\n        device = self.device(name="rtr-nyc", role="router", site_name="nyc")\n        self.add(device)\n        site.add_child(device)\n```\n\n## Update remote system on sync\n\nWhen data synchronization is performed via `sync_from()` or `sync_to()`, DiffSync automatically updates the in-memory\n`DiffSyncModel` objects of the receiving adapter. The implementer of this class is responsible for ensuring that any remote system or data store is updated correspondingly. There are two usual ways to do this, depending on whether it\'s more\nconvenient to manage individual records (as in a database) or modify the entire data store in one pass (as in a file-based data store).\n\n### Manage individual records\n\nTo update individual records in a remote system, you need to extend your `DiffSyncModel` class(es) to define your own `create`, `update` and/or `delete` methods for each model.\nA `DiffSyncModel` instance stores a reference to its parent `DiffSync` adapter instance in case you need to use it to look up other model instances from the `DiffSync`\'s cache.\n\n```python\nclass Device(DiffSyncModel):\n    [...]\n\n    @classmethod\n    def create(cls, diffsync, ids, attrs):\n        ## TODO add your own logic here to create the device on the remote system\n        # Call the super().create() method to create the in-memory DiffSyncModel instance\n        return super().create(ids=ids, diffsync=diffsync, attrs=attrs)\n\n    def update(self, attrs):\n        ## TODO add your own logic here to update the device on the remote system\n        # Call the super().update() method to update the in-memory DiffSyncModel instance\n        return super().update(attrs)\n\n    def delete(self):\n        ## TODO add your own logic here to delete the device on the remote system\n        # Call the super().delete() method to remove the DiffSyncModel instance from its parent DiffSync adapter\n        super().delete()\n        return self\n```\n\n### Bulk/batch modifications\n\nIf you prefer to update the entire remote system with the final state after performing all individual create/update/delete operations (as might be the case if your "remote system" is a single YAML or JSON file), the easiest place to implement this logic is in the `sync_complete()` callback method that is automatically invoked by DiffSync upon completion of a sync operation.\n\n```python\nclass BackendA(DiffSync):\n    [...]\n\n    def sync_complete(self, source: DiffSync, diff: Diff, flags: DiffSyncFlags, logger: structlog.BoundLogger):\n        ## TODO add your own logic to update the remote system now.\n        # The various parameters passed to this method are for your convenience in implementing more complex logic, and\n        # can be ignored if you do not need them.\n        #\n        # The default DiffSync.sync_complete() method does nothing, but it\'s always a good habit to call super():\n        super().sync_complete(source, diff, flags, logger)\n```\n',
    'author': 'Network to Code, LLC',
    'author_email': 'info@networktocode.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/networktocode/diffsync',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
