"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageJsonManager = void 0;
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * A package.json manager to act on the closest package.json file.
 *
 * Configuring the bundler requires to manipulate the package.json and then
 * restore it.
 */
class PackageJsonManager {
    constructor(directory) {
        const pkgPath = util_1.findUp('package.json', directory);
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        this.pkgPath = path.join(pkgPath, 'package.json');
        this.pkg = fs.readFileSync(this.pkgPath);
        this.pkgJson = JSON.parse(this.pkg.toString());
    }
    /**
     * Update the package.json
     */
    update(data) {
        fs.writeFileSync(this.pkgPath, JSON.stringify({
            ...this.pkgJson,
            ...data,
        }, null, 2));
    }
    /**
     * Restore the package.json to the original
     */
    restore() {
        fs.writeFileSync(this.pkgPath, this.pkg);
    }
    /**
     * Extract versions for a list of modules
     */
    getVersions(modules) {
        var _a, _b, _c;
        const dependencies = {};
        const allDependencies = {
            ...(_a = this.pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
            ...(_b = this.pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
            ...(_c = this.pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
        };
        for (const mod of modules) {
            if (!allDependencies[mod]) {
                throw new Error(`Cannot extract version for ${mod} in package.json`);
            }
            dependencies[mod] = allDependencies[mod];
        }
        return dependencies;
    }
}
exports.PackageJsonManager = PackageJsonManager;
//# sourceMappingURL=data:application/json;base64,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