"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const fs = require("fs");
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const bundling_1 = require("../lib/bundling");
jest.mock('../lib/bundling', () => {
    return {
        Bundling: {
            parcel: jest.fn().mockReturnValue({
                bind: () => {
                    return { inlineCode: 'code' };
                },
                bindToResource: () => { return; },
            }),
        },
    };
});
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
    jest.clearAllMocks();
});
test('NodejsFunction with .ts handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1');
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler1.ts'),
    }));
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
    });
});
test('NodejsFunction with .js handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler2');
    // THEN
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler2.js'),
    }));
});
test('NodejsFunction with container env vars', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1', {
        parcelEnvironment: {
            KEY: 'VALUE',
        },
    });
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        parcelEnvironment: {
            KEY: 'VALUE',
        },
    }));
});
test('throws when entry is not js/ts', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'handler.py',
    })).toThrow(/Only JavaScript or TypeScript entry files are supported/);
});
test('accepts tsx', () => {
    const entry = path.join(__dirname, 'handler.tsx');
    fs.symlinkSync(path.join(__dirname, 'function.test.handler1.ts'), entry);
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry,
    })).not.toThrow();
    fs.unlinkSync(entry);
});
test('throws when entry does not exist', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'notfound.ts',
    })).toThrow(/Cannot find entry file at/);
});
test('throws when entry cannot be automatically found', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn')).toThrow(/Cannot find entry file./);
});
test('throws with the wrong runtime family', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'handler1', {
        runtime: aws_lambda_1.Runtime.PYTHON_3_8,
    })).toThrow(/Only `NODEJS` runtimes are supported/);
});
test('resolves entry to an absolute path', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'fn', {
        entry: 'lib/index.ts',
    });
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringMatching(/@aws-cdk\/aws-lambda-nodejs\/lib\/index.ts$/),
    }));
});
test('configures connection reuse for aws sdk', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1');
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
            },
        },
    });
});
test('can opt-out of connection reuse for aws sdk', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1', {
        awsSdkConnectionReuse: false,
    });
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Environment: assert_1.ABSENT,
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZnVuY3Rpb24udGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImZ1bmN0aW9uLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxnQ0FBOEI7QUFDOUIseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3Qiw0Q0FBeUM7QUFDekMsb0RBQThDO0FBQzlDLHdDQUFzQztBQUN0QyxnQ0FBd0M7QUFDeEMsOENBQTJDO0FBRTNDLElBQUksQ0FBQyxJQUFJLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxFQUFFO0lBQ2hDLE9BQU87UUFDTCxRQUFRLEVBQUU7WUFDUixNQUFNLEVBQUUsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLGVBQWUsQ0FBQztnQkFDaEMsSUFBSSxFQUFFLEdBQUcsRUFBRTtvQkFDVCxPQUFPLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxDQUFDO2dCQUNoQyxDQUFDO2dCQUNELGNBQWMsRUFBRSxHQUFHLEVBQUUsR0FBRyxPQUFPLENBQUMsQ0FBQzthQUNsQyxDQUFDO1NBQ0g7S0FDRixDQUFDO0FBQ0osQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLEtBQVksQ0FBQztBQUNqQixVQUFVLENBQUMsR0FBRyxFQUFFO0lBQ2QsS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFDcEIsSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDO0FBQ3ZCLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtJQUMzQyxPQUFPO0lBQ1AsSUFBSSxvQkFBYyxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztJQUV0QyxNQUFNLENBQUMsbUJBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUM7UUFDbkUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQywyQkFBMkIsQ0FBQztLQUM1RCxDQUFDLENBQUMsQ0FBQztJQUVKLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsdUJBQXVCLEVBQUU7UUFDcEQsT0FBTyxFQUFFLGVBQWU7S0FDekIsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsaUNBQWlDLEVBQUUsR0FBRyxFQUFFO0lBQzNDLE9BQU87SUFDUCxJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBRXRDLE9BQU87SUFDUCxNQUFNLENBQUMsbUJBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUM7UUFDbkUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQywyQkFBMkIsQ0FBQztLQUM1RCxDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEdBQUcsRUFBRTtJQUNsRCxPQUFPO0lBQ1AsSUFBSSxvQkFBYyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7UUFDcEMsaUJBQWlCLEVBQUU7WUFDakIsR0FBRyxFQUFFLE9BQU87U0FDYjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxtQkFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQztRQUNuRSxpQkFBaUIsRUFBRTtZQUNqQixHQUFHLEVBQUUsT0FBTztTQUNiO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFHLEVBQUU7SUFDMUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1FBQzNDLEtBQUssRUFBRSxZQUFZO0tBQ3BCLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyx5REFBeUQsQ0FBQyxDQUFDO0FBQ3pFLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGFBQWEsRUFBRSxHQUFHLEVBQUU7SUFDdkIsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsYUFBYSxDQUFDLENBQUM7SUFFbEQsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwyQkFBMkIsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXpFLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtRQUMzQyxLQUFLO0tBQ04sQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRSxDQUFDO0lBRWxCLEVBQUUsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDdkIsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsa0NBQWtDLEVBQUUsR0FBRyxFQUFFO0lBQzVDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtRQUMzQyxLQUFLLEVBQUUsYUFBYTtLQUNyQixDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsMkJBQTJCLENBQUMsQ0FBQztBQUMzQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxpREFBaUQsRUFBRSxHQUFHLEVBQUU7SUFDM0QsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMseUJBQXlCLENBQUMsQ0FBQztBQUNuRixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzQ0FBc0MsRUFBRSxHQUFHLEVBQUU7SUFDaEQsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQ2pELE9BQU8sRUFBRSxvQkFBTyxDQUFDLFVBQVU7S0FDNUIsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLHNDQUFzQyxDQUFDLENBQUM7QUFDdEQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsb0NBQW9DLEVBQUUsR0FBRyxFQUFFO0lBQzlDLE9BQU87SUFDUCxJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtRQUM5QixLQUFLLEVBQUUsY0FBYztLQUN0QixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsbUJBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUM7UUFDbkUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxjQUFjLENBQUMsNkNBQTZDLENBQUM7S0FDNUUsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx5Q0FBeUMsRUFBRSxHQUFHLEVBQUU7SUFDbkQsT0FBTztJQUNQLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFFdEMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyx1QkFBdUIsRUFBRTtRQUNwRCxXQUFXLEVBQUU7WUFDWCxTQUFTLEVBQUU7Z0JBQ1QsbUNBQW1DLEVBQUUsR0FBRzthQUN6QztTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNkNBQTZDLEVBQUUsR0FBRyxFQUFFO0lBQ3ZELE9BQU87SUFDUCxJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtRQUNwQyxxQkFBcUIsRUFBRSxLQUFLO0tBQzdCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsdUJBQXVCLEVBQUU7UUFDcEQsV0FBVyxFQUFFLGVBQU07S0FDcEIsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBBQlNFTlQgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0IHsgUnVudGltZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IE5vZGVqc0Z1bmN0aW9uIH0gZnJvbSAnLi4vbGliJztcbmltcG9ydCB7IEJ1bmRsaW5nIH0gZnJvbSAnLi4vbGliL2J1bmRsaW5nJztcblxuamVzdC5tb2NrKCcuLi9saWIvYnVuZGxpbmcnLCAoKSA9PiB7XG4gIHJldHVybiB7XG4gICAgQnVuZGxpbmc6IHtcbiAgICAgIHBhcmNlbDogamVzdC5mbigpLm1vY2tSZXR1cm5WYWx1ZSh7XG4gICAgICAgIGJpbmQ6ICgpID0+IHtcbiAgICAgICAgICByZXR1cm4geyBpbmxpbmVDb2RlOiAnY29kZScgfTtcbiAgICAgICAgfSxcbiAgICAgICAgYmluZFRvUmVzb3VyY2U6ICgpID0+IHsgcmV0dXJuOyB9LFxuICAgICAgfSksXG4gICAgfSxcbiAgfTtcbn0pO1xuXG5sZXQgc3RhY2s6IFN0YWNrO1xuYmVmb3JlRWFjaCgoKSA9PiB7XG4gIHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGplc3QuY2xlYXJBbGxNb2NrcygpO1xufSk7XG5cbnRlc3QoJ05vZGVqc0Z1bmN0aW9uIHdpdGggLnRzIGhhbmRsZXInLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgbmV3IE5vZGVqc0Z1bmN0aW9uKHN0YWNrLCAnaGFuZGxlcjEnKTtcblxuICBleHBlY3QoQnVuZGxpbmcucGFyY2VsKS50b0hhdmVCZWVuQ2FsbGVkV2l0aChleHBlY3Qub2JqZWN0Q29udGFpbmluZyh7XG4gICAgZW50cnk6IGV4cGVjdC5zdHJpbmdDb250YWluaW5nKCdmdW5jdGlvbi50ZXN0LmhhbmRsZXIxLnRzJyksIC8vIEF1dG9tYXRpY2FsbHkgZmluZHMgLnRzIGhhbmRsZXIgZmlsZVxuICB9KSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywge1xuICAgIEhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgfSk7XG59KTtcblxudGVzdCgnTm9kZWpzRnVuY3Rpb24gd2l0aCAuanMgaGFuZGxlcicsICgpID0+IHtcbiAgLy8gV0hFTlxuICBuZXcgTm9kZWpzRnVuY3Rpb24oc3RhY2ssICdoYW5kbGVyMicpO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KEJ1bmRsaW5nLnBhcmNlbCkudG9IYXZlQmVlbkNhbGxlZFdpdGgoZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgIGVudHJ5OiBleHBlY3Quc3RyaW5nQ29udGFpbmluZygnZnVuY3Rpb24udGVzdC5oYW5kbGVyMi5qcycpLCAvLyBBdXRvbWF0aWNhbGx5IGZpbmRzIC50cyBoYW5kbGVyIGZpbGVcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ05vZGVqc0Z1bmN0aW9uIHdpdGggY29udGFpbmVyIGVudiB2YXJzJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIG5ldyBOb2RlanNGdW5jdGlvbihzdGFjaywgJ2hhbmRsZXIxJywge1xuICAgIHBhcmNlbEVudmlyb25tZW50OiB7XG4gICAgICBLRVk6ICdWQUxVRScsXG4gICAgfSxcbiAgfSk7XG5cbiAgZXhwZWN0KEJ1bmRsaW5nLnBhcmNlbCkudG9IYXZlQmVlbkNhbGxlZFdpdGgoZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgIHBhcmNlbEVudmlyb25tZW50OiB7XG4gICAgICBLRVk6ICdWQUxVRScsXG4gICAgfSxcbiAgfSkpO1xufSk7XG5cbnRlc3QoJ3Rocm93cyB3aGVuIGVudHJ5IGlzIG5vdCBqcy90cycsICgpID0+IHtcbiAgZXhwZWN0KCgpID0+IG5ldyBOb2RlanNGdW5jdGlvbihzdGFjaywgJ0ZuJywge1xuICAgIGVudHJ5OiAnaGFuZGxlci5weScsXG4gIH0pKS50b1Rocm93KC9Pbmx5IEphdmFTY3JpcHQgb3IgVHlwZVNjcmlwdCBlbnRyeSBmaWxlcyBhcmUgc3VwcG9ydGVkLyk7XG59KTtcblxudGVzdCgnYWNjZXB0cyB0c3gnLCAoKSA9PiB7XG4gIGNvbnN0IGVudHJ5ID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJ2hhbmRsZXIudHN4Jyk7XG5cbiAgZnMuc3ltbGlua1N5bmMocGF0aC5qb2luKF9fZGlybmFtZSwgJ2Z1bmN0aW9uLnRlc3QuaGFuZGxlcjEudHMnKSwgZW50cnkpO1xuXG4gIGV4cGVjdCgoKSA9PiBuZXcgTm9kZWpzRnVuY3Rpb24oc3RhY2ssICdGbicsIHtcbiAgICBlbnRyeSxcbiAgfSkpLm5vdC50b1Rocm93KCk7XG5cbiAgZnMudW5saW5rU3luYyhlbnRyeSk7XG59KTtcblxudGVzdCgndGhyb3dzIHdoZW4gZW50cnkgZG9lcyBub3QgZXhpc3QnLCAoKSA9PiB7XG4gIGV4cGVjdCgoKSA9PiBuZXcgTm9kZWpzRnVuY3Rpb24oc3RhY2ssICdGbicsIHtcbiAgICBlbnRyeTogJ25vdGZvdW5kLnRzJyxcbiAgfSkpLnRvVGhyb3coL0Nhbm5vdCBmaW5kIGVudHJ5IGZpbGUgYXQvKTtcbn0pO1xuXG50ZXN0KCd0aHJvd3Mgd2hlbiBlbnRyeSBjYW5ub3QgYmUgYXV0b21hdGljYWxseSBmb3VuZCcsICgpID0+IHtcbiAgZXhwZWN0KCgpID0+IG5ldyBOb2RlanNGdW5jdGlvbihzdGFjaywgJ0ZuJykpLnRvVGhyb3coL0Nhbm5vdCBmaW5kIGVudHJ5IGZpbGUuLyk7XG59KTtcblxudGVzdCgndGhyb3dzIHdpdGggdGhlIHdyb25nIHJ1bnRpbWUgZmFtaWx5JywgKCkgPT4ge1xuICBleHBlY3QoKCkgPT4gbmV3IE5vZGVqc0Z1bmN0aW9uKHN0YWNrLCAnaGFuZGxlcjEnLCB7XG4gICAgcnVudGltZTogUnVudGltZS5QWVRIT05fM184LFxuICB9KSkudG9UaHJvdygvT25seSBgTk9ERUpTYCBydW50aW1lcyBhcmUgc3VwcG9ydGVkLyk7XG59KTtcblxudGVzdCgncmVzb2x2ZXMgZW50cnkgdG8gYW4gYWJzb2x1dGUgcGF0aCcsICgpID0+IHtcbiAgLy8gV0hFTlxuICBuZXcgTm9kZWpzRnVuY3Rpb24oc3RhY2ssICdmbicsIHtcbiAgICBlbnRyeTogJ2xpYi9pbmRleC50cycsXG4gIH0pO1xuXG4gIGV4cGVjdChCdW5kbGluZy5wYXJjZWwpLnRvSGF2ZUJlZW5DYWxsZWRXaXRoKGV4cGVjdC5vYmplY3RDb250YWluaW5nKHtcbiAgICBlbnRyeTogZXhwZWN0LnN0cmluZ01hdGNoaW5nKC9AYXdzLWNka1xcL2F3cy1sYW1iZGEtbm9kZWpzXFwvbGliXFwvaW5kZXgudHMkLyksXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdjb25maWd1cmVzIGNvbm5lY3Rpb24gcmV1c2UgZm9yIGF3cyBzZGsnLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgbmV3IE5vZGVqc0Z1bmN0aW9uKHN0YWNrLCAnaGFuZGxlcjEnKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCB7XG4gICAgRW52aXJvbm1lbnQ6IHtcbiAgICAgIFZhcmlhYmxlczoge1xuICAgICAgICBBV1NfTk9ERUpTX0NPTk5FQ1RJT05fUkVVU0VfRU5BQkxFRDogJzEnLFxuICAgICAgfSxcbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdjYW4gb3B0LW91dCBvZiBjb25uZWN0aW9uIHJldXNlIGZvciBhd3Mgc2RrJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIG5ldyBOb2RlanNGdW5jdGlvbihzdGFjaywgJ2hhbmRsZXIxJywge1xuICAgIGF3c1Nka0Nvbm5lY3Rpb25SZXVzZTogZmFsc2UsXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICBFbnZpcm9ubWVudDogQUJTRU5ULFxuICB9KTtcbn0pO1xuIl19