"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiSpecProject = void 0;
const path = require("path");
const fs_extra_1 = require("fs-extra");
const lodash_1 = require("lodash");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
const parsed_spec_1 = require("./components/parsed-spec");
// initialize logger
const logger = log4js_1.getLogger();
/**
 * Project containing the OpenAPI spec, and a parsed spec for use by the CDK construct
 */
class OpenApiSpecProject extends projen_1.Project {
    constructor(options) {
        super(options);
        this.specChanged = true;
        // Store whether we've synthesized the project
        this.synthed = false;
        logger.trace("OpenApiSpecProject constructor");
        // HACK: remove all components but the ones we are registering - removes .gitignore, tasks, etc since these are
        // unused and a distraction for end-users!
        // @ts-ignore
        this._components = [];
        this.specFileName = options.specFileName ?? "spec.yaml";
        this.parsedSpecFileName = options.parsedSpecFileName ?? ".parsed-spec.json";
        if (!this.parsedSpecFileName.endsWith(".json")) {
            throw new Error("Parsed spec file must end with .json");
        }
        this.specPath = path.join(this.outdir, this.specFileName);
        this.parsedSpecPath = path.join(this.outdir, ...(options.parsedSpecOutdir ? [options.parsedSpecOutdir] : []), this.parsedSpecFileName);
        logger.debug(`specPath = "${this.specPath}"`);
        logger.debug(`parsedSpecPath = "${this.parsedSpecPath}"`);
        // Create a sample OpenAPI spec yaml if not defined
        new projen_1.SampleFile(this, this.specFileName, {
            sourcePath: path.join(__dirname, "..", "..", "..", "samples", "spec.yaml"),
        });
        // Check if there is already a .parsed-spec.json present
        const existingParsedSpecJson = util_1.tryReadFileSync(this.parsedSpecPath);
        if (existingParsedSpecJson != null) {
            // generate a new temporary parsed-spec
            const tmpParseSpecPath = `${this.parsedSpecPath}.tmp`;
            logger.trace(`Generating temp spec at "${tmpParseSpecPath}"`);
            parsed_spec_1.ParsedSpec.parse(this.specPath, tmpParseSpecPath);
            // load it
            const newParsedSpecJson = util_1.tryReadFileSync(tmpParseSpecPath);
            if (newParsedSpecJson != null) {
                const prevParseSpec = JSON.parse(existingParsedSpecJson);
                const newParsedSpec = JSON.parse(newParsedSpecJson);
                // check if spec changed and keep it
                logger.trace("Comparing previous and newly generated specs.");
                this.specChanged = !lodash_1.isEqual(prevParseSpec, newParsedSpec);
                logger.debug(`Spec Changed :: ${this.specChanged}`);
                // remove tmp parsed-spec file
                logger.trace("Removing temp spec file.");
                fs_extra_1.unlinkSync(tmpParseSpecPath);
            }
        }
        else {
            logger.debug(`No parsedSpec file found at "${this.parsedSpecPath}".`);
        }
        // Parse the spec to produce a consolidated, bundled spec which can be read by cdk constructs or other tooling
        new parsed_spec_1.ParsedSpec(this, {
            specPath: this.specPath,
            outputPath: this.parsedSpecPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            logger.trace("OpenApiSpecProject already synthed. Skipping...");
            return;
        }
        logger.trace("OpenApiSpecProject synth");
        super.synth();
        this.synthed = true;
    }
}
exports.OpenApiSpecProject = OpenApiSpecProject;
//# sourceMappingURL=data:application/json;base64,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