"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayLambdaApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const utils_1 = require("./spec/utils");
const open_api_gateway_web_acl_1 = require("./waf/open-api-gateway-web-acl");
/**
 * A construct for creating an api gateway api based on the definition in the OpenAPI spec.
 */
class OpenApiGatewayLambdaApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, spec, specPath, operationLookup, defaultAuthorizer, corsOptions, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKeyPrefix = `${inputSpecAsset.s3ObjectKey}-prepared`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        const prepareSpecLambdaName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}PrepareSpec`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                // The output file will include a hash of the prepared spec, which is not known until deploy time since
                                // tokens must be resolved
                                inputSpecAsset.bucket.arnForObjects(`${preparedSpecOutputKeyPrefix}/*`),
                            ],
                        }),
                    ],
                }),
            },
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                    },
                ],
            },
        ], true);
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpec", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../../lib/construct/prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${prepareSpecLambdaName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
            {
                id: "AwsSolutions-L1",
                reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
            },
        ], true);
        const prepareSpecOptions = {
            defaultAuthorizerReference: api_gateway_auth_1.serializeAsAuthorizerReference(defaultAuthorizer),
            integrations: Object.fromEntries(Object.entries(integrations).map(([operation, integration]) => [
                operation,
                {
                    functionInvocationUri: utils_1.functionInvocationUri(this, integration.function),
                    methodAuthorizer: api_gateway_auth_1.serializeAsAuthorizerReference(integration.authorizer),
                },
            ])),
            securitySchemes: api_gateway_auth_1.prepareSecuritySchemes(this, integrations, defaultAuthorizer),
            corsOptions: corsOptions && {
                allowHeaders: corsOptions.allowHeaders || aws_apigateway_1.Cors.DEFAULT_HEADERS,
                allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
                allowOrigins: corsOptions.allowOrigins,
                statusCode: corsOptions.statusCode || 204,
            },
            operationLookup,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        const preparedSpec = prepare_spec_1.prepareApiSpec(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: preparedSpecOutputKeyPrefix,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecCustomResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: aws_apigateway_1.ApiDefinition.fromBucket(inputSpecAsset.bucket, prepareSpecCustomResource.getAttString("outputSpecKey")),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(scope, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // While the api will be updated when the output path from the custom resource changes, CDK still needs to know when
        // to redeploy the api. This is achieved by including a hash of the spec in the logical id (internalised in the
        // addToLogicalId method since this is how changes of individual resources/methods etc trigger redeployments in CDK)
        this.api.latestDeployment?.addToLogicalId(preparedSpec);
        // Grant API Gateway permission to invoke each lambda which implements an integration or custom authorizer
        api_gateway_integrations_1.getLabelledFunctions(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        // Create and associate the web acl if not disabled
        if (!props.webAclOptions?.disable) {
            const acl = new open_api_gateway_web_acl_1.OpenApiGatewayWebAcl(this, `${id}-Acl`, {
                ...props.webAclOptions,
                apiDeploymentStageArn: this.api.deploymentStage.stageArn,
            });
            this.webAcl = acl.webAcl;
            this.ipSet = acl.ipSet;
            this.webAclAssociation = acl.webAclAssociation;
        }
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-IAM4",
                reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                appliesTo: [
                    {
                        regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                    },
                ],
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-APIG2",
                reason: "This construct implements fine grained validation via OpenApi.",
            },
        ], true);
    }
}
exports.OpenApiGatewayLambdaApi = OpenApiGatewayLambdaApi;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayLambdaApi[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayLambdaApi", version: "0.10.3" };
//# sourceMappingURL=data:application/json;base64,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