"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getJavaSampleSource = void 0;
exports.getJavaSampleSource = (options) => {
    const apiPackage = options.apiSrcDir.split("/").join(".");
    return {
        "ApiProps.java": `package ${apiPackage};

import software.amazon.awscdk.services.apigateway.CorsOptions;
import software.amazon.awscdk.services.apigateway.StageOptions;
import software.amazon.awscdk.services.apigateway.RestApiBaseProps;
import software.amazon.awscdk.services.apigateway.DomainNameOptions;
import software.amazon.awscdk.services.apigateway.EndpointType;
import software.amazon.awscdk.services.iam.PolicyDocument;
import software.aws.awsprototypingsdk.openapigateway.Authorizer;
import software.aws.awsprototypingsdk.openapigateway.OpenApiIntegration;

import ${options.javaClientPackage}.api.OperationConfig;

import java.util.List;
import java.util.Map;

/**
 * Properties for the Api construct
 */
@lombok.Builder @lombok.Getter
public class ApiProps implements RestApiBaseProps {
    public OperationConfig<OpenApiIntegration> integrations;
    public Authorizer defaultAuthorizer;
    public CorsOptions corsOptions;

    // Rest API Props
    public Boolean cloudWatchRole;
    public Boolean deploy;
    public StageOptions deployOptions;
    public String description;
    public Boolean disableExecuteApiEndpoint;
    public DomainNameOptions domainName;
    public String endpointExportName;
    public List<EndpointType> endpointTypes;
    public Boolean failOnWarnings;
    public Map<String, String> parameters;
    public PolicyDocument policy;
    public String restApiName;
    public Boolean retainDeployments;
}
`,
        "Api.java": `package ${apiPackage};

import com.fasterxml.jackson.databind.ObjectMapper;
import software.aws.awsprototypingsdk.openapigateway.MethodAndPath;
import software.aws.awsprototypingsdk.openapigateway.OpenApiGatewayLambdaApi;
import software.aws.awsprototypingsdk.openapigateway.OpenApiGatewayLambdaApiProps;
import software.constructs.Construct;
import ${options.javaClientPackage}.api.OperationLookup;

import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Type-safe construct for the API Gateway resources defined by the spec.
 * You will likely not need to modify this file, and can instead extend it and define your integrations.
 */
public class Api extends OpenApiGatewayLambdaApi {
    private static class SpecDetails {
        static String specPath;
        static Object spec;

        static {
            try {
                // The parsed spec is included as a packaged resource
                URL specUrl = SpecDetails.class.getClassLoader().getResource("${options.specDir}/${options.parsedSpecFileName}");

                // We'll write the parsed spec to a temporary file outside of the jar to ensure CDK can package it as an asset
                Path parsedSpecPath = Files.createTempFile("parsed-spec", ".json");
                specPath = parsedSpecPath.toString();

                ObjectMapper json = new ObjectMapper();
                spec = json.readValue(specUrl, Object.class);
                json.writeValue(parsedSpecPath.toFile(), spec);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public Api(Construct scope, String id, ApiProps props) {
        super(scope, id, OpenApiGatewayLambdaApiProps.builder()
                .defaultAuthorizer(props.getDefaultAuthorizer())
                .corsOptions(props.getCorsOptions())
                .operationLookup(OperationLookup.getOperationLookup()
                        .entrySet()
                        .stream()
                        .collect(Collectors.toMap(Map.Entry::getKey, e -> MethodAndPath.builder()
                                .method(e.getValue().get("method"))
                                .path(e.getValue().get("path"))
                                .build())))
                .spec(SpecDetails.spec)
                .specPath(SpecDetails.specPath)
                .integrations(props.getIntegrations().asMap())
                // Rest API Base Props
                .cloudWatchRole(props.getCloudWatchRole())
                .deploy(props.getDeploy())
                .deployOptions(props.getDeployOptions())
                .description(props.getDescription())
                .disableExecuteApiEndpoint(props.getDisableExecuteApiEndpoint())
                .domainName(props.getDomainName())
                .endpointExportName(props.getEndpointExportName())
                .endpointTypes(props.getEndpointTypes())
                .failOnWarnings(props.getFailOnWarnings())
                .parameters(props.getParameters())
                .policy(props.getPolicy())
                .restApiName(props.getRestApiName())
                .retainDeployments(props.getRetainDeployments())
                .build());
    }
}
`,
        ...(options.sampleCode !== false
            ? {
                "SayHelloHandler.java": `package ${apiPackage};

import ${options.javaClientPackage}.api.Handlers.SayHello;
import ${options.javaClientPackage}.api.Handlers.SayHello200Response;
import ${options.javaClientPackage}.api.Handlers.SayHelloRequestInput;
import ${options.javaClientPackage}.api.Handlers.SayHelloResponse;
import ${options.javaClientPackage}.model.SayHelloResponseContent;

/**
 * An example lambda handler which uses the generated handler wrapper class (Handlers.SayHello) to manage marshalling
 * inputs and outputs.
 */
public class SayHelloHandler extends SayHello {
    @Override
    public SayHelloResponse handle(SayHelloRequestInput sayHelloRequestInput) {
        return SayHello200Response.of(SayHelloResponseContent.builder()
                .message(String.format("Hello %s", sayHelloRequestInput.getInput().getRequestParameters().getName()))
                .build());
    }
}
`,
                "SampleApi.java": `package ${apiPackage};

import software.amazon.awscdk.Duration;
import software.amazon.awscdk.services.apigateway.CorsOptions;
import software.amazon.awscdk.services.lambda.Code;
import software.amazon.awscdk.services.lambda.Function;
import software.amazon.awscdk.services.lambda.FunctionProps;
import software.amazon.awscdk.services.lambda.Runtime;
import software.aws.awsprototypingsdk.openapigateway.Authorizers;
import software.aws.awsprototypingsdk.openapigateway.OpenApiIntegration;
import ${options.javaClientPackage}.api.OperationConfig;
import software.constructs.Construct;

import java.net.URISyntaxException;
import java.util.Arrays;

/**
 * An example of how to wire lambda handler functions to API operations
 */
public class SampleApi extends Api {
    public SampleApi(Construct scope, String id) {
        super(scope, id, ApiProps.builder()
                .defaultAuthorizer(Authorizers.iam())
                .corsOptions(CorsOptions.builder()
                        .allowOrigins(Arrays.asList("*"))
                        .allowMethods(Arrays.asList("*"))
                        .build())
                .integrations(OperationConfig.<OpenApiIntegration>builder()
                        .sayHello(OpenApiIntegration.builder()
                                .function(new Function(scope, "say-hello", FunctionProps.builder()
                                        // Use the entire project jar for the lambda code in order to provide a simple,
                                        // "one-click" way to build the api. However this jar is much larger than necessary
                                        // since it includes cdk infrastructure, dependencies etc.
                                        // It is recommended to follow the instructions in the "Java API Lambda Handlers"
                                        // section of the open-api-gateway README to define your lambda handlers as a
                                        // separate project.
                                        .code(Code.fromAsset(SampleApi.getJarPath()))
                                        .handler("${apiPackage}.SayHelloHandler")
                                        .runtime(Runtime.JAVA_11)
                                        .timeout(Duration.seconds(30))
                                        .build()))
                                .build())
                        .build())
                .build());
    }

    private static String getJarPath() {
        try {
            // Retrieve the path of the jar in which this class resides
            return SampleApi.class.getProtectionDomain().getCodeSource().getLocation().toURI().getPath();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }
}

`,
            }
            : {}),
    };
};
//# sourceMappingURL=data:application/json;base64,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