"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPythonSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample python source code
 */
exports.getPythonSampleSource = (options) => ({
    "__init__.py": "#",
    // This file provides a type-safe interface to the exported OpenApiGatewayLambdaApi construct
    "api.py": `from dataclasses import fields
from ${options.openApiGatewayPackageName} import OpenApiGatewayLambdaApi, OpenApiIntegration
from ${options.pythonClientPackageName}.apis.tags.default_api_operation_config import OperationLookup, OperationConfig
from ${options.moduleName}.spec_utils import SPEC, SPEC_PATH

class Api(OpenApiGatewayLambdaApi):
    """
    Type-safe construct for the API Gateway resources defined by the spec.
    You will likely not need to modify this file, and can instead extend it and define your integrations.
    """
    def __init__(self, scope, id, integrations: OperationConfig[OpenApiIntegration], **kwargs):
        super().__init__(scope, id,
            **kwargs,
            integrations={ field.name: getattr(integrations, field.name) for field in fields(integrations) },
            spec=SPEC,
            spec_path=SPEC_PATH,
            operation_lookup=OperationLookup,
        )
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample_api.py": `from ${options.openApiGatewayPackageName} import Authorizers, OpenApiIntegration
from ${options.pythonClientPackageName}.apis.tags.default_api_operation_config import OperationConfig
from aws_cdk.aws_lambda import LayerVersion, Code, Function, Runtime
from .api import Api
from constructs import Construct
from ${options.moduleName}.spec_utils import get_generated_client_layer_directory
from pathlib import Path
from os import path

class SampleApi(Construct):
  """
  An example of how to add integrations to your api
  """
  def __init__(self, scope, id):
      super().__init__(scope, id)

      # Layer which contains the generated client.
      self.generated_client_layer = LayerVersion(self, 'GeneratedClientLayer',
          code=Code.from_asset(get_generated_client_layer_directory())
      )

      self.api = Api(self, 'Api',
          default_authorizer=Authorizers.iam(),
          integrations=OperationConfig(
              say_hello=OpenApiIntegration(
                  function=Function(self, 'SayHello',
                      runtime=Runtime.PYTHON_3_9,
                      code=Code.from_asset(path.join(str(Path(__file__).parent.absolute()), 'handlers')),
                      handler="say_hello_handler_sample.handler",
                      layers=[self.generated_client_layer],
                  ),
              ),
          ),
      )
`,
            // Generate an example lambda handler
            "handlers/say_hello_handler_sample.py": `from ${options.pythonClientPackageName}.apis.tags.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse, SayHelloOperationResponses
from ${options.pythonClientPackageName}.model.hello_response import SayHelloResponseContent

@say_hello_handler
def handler(input: SayHelloRequest, **kwargs) -> SayHelloOperationResponses:
    """
    An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs
    """
    return ApiResponse(
        status_code=200,
        body=SayHelloResponseContent(message="Hello {}!".format(input.request_parameters["name"])),
        headers={}
    )
`,
        }
        : {}),
});
//# sourceMappingURL=data:application/json;base64,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