"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoEventSource = void 0;
const stream_1 = require("./stream");
/**
 * Use an Amazon DynamoDB stream as an event source for AWS Lambda.
 */
class DynamoEventSource extends stream_1.StreamEventSource {
    /**
     *
     */
    constructor(table, props) {
        super(props);
        this.table = table;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 1000)) {
            throw new Error(`Maximum batch size must be between 1 and 1000 inclusive (given ${this.props.batchSize})`);
        }
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     */
    bind(target) {
        if (!this.table.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.table.node.path}`);
        }
        const eventSourceMapping = target.addEventSourceMapping(`DynamoDBEventSource:${this.table.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.table.tableStreamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.table.grantStreamRead(target);
    }
    /**
     * The identifier for this EventSourceMapping.
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('DynamoEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.DynamoEventSource = DynamoEventSource;
//# sourceMappingURL=data:application/json;base64,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