"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisEventSource = void 0;
const cdk = require("@aws-cdk/core");
const stream_1 = require("./stream");
/**
 * Use an Amazon Kinesis stream as an event source for AWS Lambda.
 */
class KinesisEventSource extends stream_1.StreamEventSource {
    /**
     *
     */
    constructor(stream, props) {
        super(props);
        this.stream = stream;
        this._eventSourceMappingId = undefined;
        this.props.batchSize !== undefined && cdk.withResolved(this.props.batchSize, batchSize => {
            if (batchSize < 1 || batchSize > 10000) {
                throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize})`);
            }
        });
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     */
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`KinesisEventSource:${this.stream.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.stream.streamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.stream.grantRead(target);
        // The `grantRead` API provides all the permissions recommended by the Kinesis team for reading a stream.
        // `DescribeStream` permissions are not required to read a stream as it's covered by the `DescribeStreamSummary`
        // and `SubscribeToShard` APIs.
        // The Lambda::EventSourceMapping resource validates against the `DescribeStream` permission. So we add it explicitly.
        // FIXME This permission can be removed when the event source mapping resource drops it from validation.
        this.stream.grant(target, 'kinesis:DescribeStream');
    }
    /**
     * The identifier for this EventSourceMapping.
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KinesisEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.KinesisEventSource = KinesisEventSource;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2luZXNpcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImtpbmVzaXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBRUEscUNBQXFDO0FBQ3JDLHFDQUFxRTs7OztBQVFyRSxNQUFhLGtCQUFtQixTQUFRLDBCQUFpQjs7OztJQUd2RCxZQUFxQixNQUF1QixFQUFFLEtBQThCO1FBQzFFLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQURNLFdBQU0sR0FBTixNQUFNLENBQWlCO1FBRnBDLDBCQUFxQixHQUFZLFNBQVMsQ0FBQztRQUtqRCxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUMsRUFBRTtZQUN2RixJQUFJLFNBQVMsR0FBRyxDQUFDLElBQUksU0FBUyxHQUFHLEtBQUssRUFBRTtnQkFDdEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRUFBbUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLEdBQUcsQ0FBQyxDQUFDO2FBQzdHO1FBQ0gsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDOzs7O0lBRU0sSUFBSSxDQUFDLE1BQXdCO1FBQ2xDLE1BQU0sa0JBQWtCLEdBQUcsTUFBTSxDQUFDLHFCQUFxQixDQUFDLHNCQUFzQixJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsRUFDdkcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEVBQUUsY0FBYyxFQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FDckUsQ0FBQztRQUNGLElBQUksQ0FBQyxxQkFBcUIsR0FBRyxrQkFBa0IsQ0FBQyxvQkFBb0IsQ0FBQztRQUVyRSxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUU5Qix5R0FBeUc7UUFDekcsZ0hBQWdIO1FBQ2hILCtCQUErQjtRQUMvQixzSEFBc0g7UUFDdEgsd0dBQXdHO1FBQ3hHLElBQUksQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO0lBQ3RELENBQUM7Ozs7SUFLRCxJQUFXLG9CQUFvQjtRQUM3QixJQUFJLENBQUMsSUFBSSxDQUFDLHFCQUFxQixFQUFFO1lBQy9CLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztTQUNuRjtRQUNELE9BQU8sSUFBSSxDQUFDLHFCQUFxQixDQUFDO0lBQ3BDLENBQUM7Q0FDRjtBQXRDRCxnREFzQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBraW5lc2lzIGZyb20gJ0Bhd3MtY2RrL2F3cy1raW5lc2lzJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFN0cmVhbUV2ZW50U291cmNlLCBTdHJlYW1FdmVudFNvdXJjZVByb3BzIH0gZnJvbSAnLi9zdHJlYW0nO1xuXG5leHBvcnQgaW50ZXJmYWNlIEtpbmVzaXNFdmVudFNvdXJjZVByb3BzIGV4dGVuZHMgU3RyZWFtRXZlbnRTb3VyY2VQcm9wcyB7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgS2luZXNpc0V2ZW50U291cmNlIGV4dGVuZHMgU3RyZWFtRXZlbnRTb3VyY2Uge1xuICBwcml2YXRlIF9ldmVudFNvdXJjZU1hcHBpbmdJZD86IHN0cmluZyA9IHVuZGVmaW5lZDtcblxuICBjb25zdHJ1Y3RvcihyZWFkb25seSBzdHJlYW06IGtpbmVzaXMuSVN0cmVhbSwgcHJvcHM6IEtpbmVzaXNFdmVudFNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuXG4gICAgdGhpcy5wcm9wcy5iYXRjaFNpemUgIT09IHVuZGVmaW5lZCAmJiBjZGsud2l0aFJlc29sdmVkKHRoaXMucHJvcHMuYmF0Y2hTaXplLCBiYXRjaFNpemUgPT4ge1xuICAgICAgaWYgKGJhdGNoU2l6ZSA8IDEgfHwgYmF0Y2hTaXplID4gMTAwMDApIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBNYXhpbXVtIGJhdGNoIHNpemUgbXVzdCBiZSBiZXR3ZWVuIDEgYW5kIDEwMDAwIGluY2x1c2l2ZSAoZ2l2ZW4gJHt0aGlzLnByb3BzLmJhdGNoU2l6ZX0pYCk7XG4gICAgICB9XG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgYmluZCh0YXJnZXQ6IGxhbWJkYS5JRnVuY3Rpb24pIHtcbiAgICBjb25zdCBldmVudFNvdXJjZU1hcHBpbmcgPSB0YXJnZXQuYWRkRXZlbnRTb3VyY2VNYXBwaW5nKGBLaW5lc2lzRXZlbnRTb3VyY2U6JHt0aGlzLnN0cmVhbS5ub2RlLnVuaXF1ZUlkfWAsXG4gICAgICB0aGlzLmVucmljaE1hcHBpbmdPcHRpb25zKHsgZXZlbnRTb3VyY2VBcm46IHRoaXMuc3RyZWFtLnN0cmVhbUFybiB9KSxcbiAgICApO1xuICAgIHRoaXMuX2V2ZW50U291cmNlTWFwcGluZ0lkID0gZXZlbnRTb3VyY2VNYXBwaW5nLmV2ZW50U291cmNlTWFwcGluZ0lkO1xuXG4gICAgdGhpcy5zdHJlYW0uZ3JhbnRSZWFkKHRhcmdldCk7XG5cbiAgICAvLyBUaGUgYGdyYW50UmVhZGAgQVBJIHByb3ZpZGVzIGFsbCB0aGUgcGVybWlzc2lvbnMgcmVjb21tZW5kZWQgYnkgdGhlIEtpbmVzaXMgdGVhbSBmb3IgcmVhZGluZyBhIHN0cmVhbS5cbiAgICAvLyBgRGVzY3JpYmVTdHJlYW1gIHBlcm1pc3Npb25zIGFyZSBub3QgcmVxdWlyZWQgdG8gcmVhZCBhIHN0cmVhbSBhcyBpdCdzIGNvdmVyZWQgYnkgdGhlIGBEZXNjcmliZVN0cmVhbVN1bW1hcnlgXG4gICAgLy8gYW5kIGBTdWJzY3JpYmVUb1NoYXJkYCBBUElzLlxuICAgIC8vIFRoZSBMYW1iZGE6OkV2ZW50U291cmNlTWFwcGluZyByZXNvdXJjZSB2YWxpZGF0ZXMgYWdhaW5zdCB0aGUgYERlc2NyaWJlU3RyZWFtYCBwZXJtaXNzaW9uLiBTbyB3ZSBhZGQgaXQgZXhwbGljaXRseS5cbiAgICAvLyBGSVhNRSBUaGlzIHBlcm1pc3Npb24gY2FuIGJlIHJlbW92ZWQgd2hlbiB0aGUgZXZlbnQgc291cmNlIG1hcHBpbmcgcmVzb3VyY2UgZHJvcHMgaXQgZnJvbSB2YWxpZGF0aW9uLlxuICAgIHRoaXMuc3RyZWFtLmdyYW50KHRhcmdldCwgJ2tpbmVzaXM6RGVzY3JpYmVTdHJlYW0nKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgZ2V0IGV2ZW50U291cmNlTWFwcGluZ0lkKCk6IHN0cmluZyB7XG4gICAgaWYgKCF0aGlzLl9ldmVudFNvdXJjZU1hcHBpbmdJZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdLaW5lc2lzRXZlbnRTb3VyY2UgaXMgbm90IHlldCBib3VuZCB0byBhbiBldmVudCBzb3VyY2UgbWFwcGluZycpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fZXZlbnRTb3VyY2VNYXBwaW5nSWQ7XG4gIH1cbn1cbiJdfQ==