import { ClientLanguage } from "../../languages";
/**
 * Enum for generator directories for non-client generators
 */
export declare enum NonClientGeneratorDirectory {
    DOCS = "docs"
}
/**
 * Generator directory for openapi generation containing templates, config etc.
 */
export declare type GeneratorDirectory = ClientLanguage | NonClientGeneratorDirectory;
/**
 * Types of normalizers supported by openapi-generator
 * @see https://openapi-generator.tech/docs/customization/#openapi-normalizer
 */
export declare type OpenApiNormalizer = "KEEP_ONLY_FIRST_TAG_IN_OPERATION";
/**
 * Options for generating client code or docs using OpenAPI Generator CLI
 */
export interface GenerationOptions {
    /**
     * The OpenAPI generator to use to generate the code/docs
     */
    readonly generator: string;
    /**
     * The directory to use for OpenAPI generation
     */
    readonly generatorDirectory: GeneratorDirectory;
    /**
     * The path of the OpenAPI spec to generate the client for
     */
    readonly specPath: string;
    /**
     * The directory in which the generated code should be output
     */
    readonly outputPath: string;
    /**
     * Additional properties to pass to the generate cli
     */
    readonly additionalProperties?: {
        [key: string]: string;
    };
    /**
     * Supply the relative path from the code project root to the source code directory in which custom generated files
     * (eg. operation config) should be placed.
     */
    readonly srcDir?: string;
    /**
     * Normalizers to apply to the spec prior to generation, if any
     * @see https://openapi-generator.tech/docs/customization/#openapi-normalizer
     */
    readonly normalizers?: Partial<Record<OpenApiNormalizer, boolean>>;
}
/**
 * Generate client code or docs by invoking the root generate script
 */
export declare const invokeOpenApiGenerator: (options: GenerationOptions) => void;
/**
 * Options for generating documentation via a custom generator script
 */
export interface CustomDocsGenerationOptions {
    /**
     * Name of the generator script which exists in scripts/custom/docs
     */
    readonly generator: string;
    /**
     * Any arguments to pass to the script
     */
    readonly args?: string;
}
/**
 * Invoke a custom documentation generator script
 */
export declare const invokeCustomDocsGenerator: (options: CustomDocsGenerationOptions) => void;
