"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stageDependencies = exports.bundle = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
function bundle(options) {
    const { entry, runtime, outputPathSuffix } = options;
    const stagedir = cdk.FileSystem.mkdtemp('python-bundling-');
    const hasDeps = stageDependencies(entry, stagedir);
    const depsCommand = chain([
        hasDeps ? `rsync -r ${exports.BUNDLER_DEPENDENCIES_CACHE}/. ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}` : '',
        `rsync -r . ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}`,
    ]);
    // Determine which dockerfile to use. When dependencies are present, we use a
    // Dockerfile that can create a cacheable layer. We can't use this Dockerfile
    // if there aren't dependencies or the Dockerfile will complain about missing
    // sources.
    const dockerfile = hasDeps
        ? 'Dockerfile.dependencies'
        : 'Dockerfile';
    // copy Dockerfile to workdir
    fs.copyFileSync(path.join(__dirname, dockerfile), path.join(stagedir, dockerfile));
    const image = cdk.DockerImage.fromBuild(stagedir, {
        buildArgs: {
            IMAGE: runtime.bundlingDockerImage.image,
        },
        file: dockerfile,
    });
    return lambda.Code.fromAsset(entry, {
        assetHashType: options.assetHashType,
        assetHash: options.assetHash,
        exclude: exports.DEPENDENCY_EXCLUDES,
        bundling: {
            image,
            command: ['bash', '-c', depsCommand],
        },
    });
}
exports.bundle = bundle;
/**
 * Checks to see if the `entry` directory contains a type of dependency that
 * we know how to install.
 */
function stageDependencies(entry, stagedir) {
    const prefixes = [
        'Pipfile',
        'pyproject',
        'poetry',
        'requirements.txt',
    ];
    let found = false;
    for (const file of fs.readdirSync(entry)) {
        for (const prefix of prefixes) {
            if (file.startsWith(prefix)) {
                fs.copyFileSync(path.join(entry, file), path.join(stagedir, file));
                found = true;
            }
        }
    }
    return found;
}
exports.stageDependencies = stageDependencies;
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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