"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPythonSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample python source code
 */
exports.getPythonSampleSource = (options) => ({
    "__init__.py": "#",
    // This file provides a type-safe interface to the exported OpenApiGatewayLambdaApi construct
    "api.py": `from dataclasses import fields
from ${options.openApiGatewayPackageName} import OpenApiGatewayLambdaApi, OpenApiIntegration
from ${options.pythonClientPackageName}.api.default_api_operation_config import OperationLookup, OperationConfig
from ${options.moduleName} import SPEC, SPEC_PATH

class Api(OpenApiGatewayLambdaApi):
    """
    Type-safe construct for the API Gateway resources defined by the spec.
    You will likely not need to modify this file, and can instead extend it and define your integrations.
    """
    def __init__(self, scope, id, integrations: OperationConfig[OpenApiIntegration], **kwargs):
        super().__init__(scope, id,
            **kwargs,
            integrations={ field.name: getattr(integrations, field.name) for field in fields(integrations) },
            spec=SPEC,
            spec_path=SPEC_PATH,
            operation_lookup=OperationLookup,
        )
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample_api.py": `from ${options.openApiGatewayPackageName} import Authorizers, OpenApiIntegration
from ${options.pythonClientPackageName}.api.default_api_operation_config import OperationConfig
from aws_cdk.aws_lambda import LayerVersion, Code, Function, Runtime
from .api import Api
from constructs import Construct
from ${options.moduleName} import get_generated_client_layer_directory
from pathlib import Path
from os import path

class SampleApi(Construct):
  """
  An example of how to add integrations to your api
  """
  def __init__(self, scope, id):
      super().__init__(scope, id)

      # Layer which contains the generated client.
      self.generated_client_layer = LayerVersion(self, 'GeneratedClientLayer',
          code=Code.from_asset(get_generated_client_layer_directory())
      )

      self.api = Api(self, 'Api',
          default_authorizer=Authorizers.iam(),
          integrations=OperationConfig(
              say_hello=OpenApiIntegration(
                  function=Function(self, 'SayHello',
                      runtime=Runtime.PYTHON_3_9,
                      code=Code.from_asset(path.join(str(Path(__file__).parent.absolute()), 'handlers')),
                      handler="say_hello_handler_sample.handler",
                      layers=[self.generated_client_layer],
                  ),
              ),
          ),
      )
`,
            // Generate an example lambda handler
            "handlers/say_hello_handler_sample.py": `from ${options.pythonClientPackageName}.api.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse
from ${options.pythonClientPackageName}.model.api_error import ApiError
from ${options.pythonClientPackageName}.model.hello_response import HelloResponse

@say_hello_handler
def handler(input: SayHelloRequest, **kwargs) -> ApiResponse[HelloResponse, ApiError]:
    """
    An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs
    """
    return ApiResponse(
        status_code=200,
        body=HelloResponse(message="Hello {}!".format(input.request_parameters["name"])),
        headers={}
    )
`,
        }
        : {}),
});
//# sourceMappingURL=data:application/json;base64,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