# The MIT License (MIT)
#
# Copyright (c) 2019 Brent Rubell for Adafruit Industries
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
"""
`adafruit_ntp`
================================================================================

Network Time Protocol (NTP) helper for CircuitPython


* Author(s): Brent Rubell

Implementation Notes
--------------------

**Hardware:**

**Software and Dependencies:**

* Adafruit CircuitPython firmware for the supported boards:
  https://github.com/adafruit/circuitpython/releases

"""
import time
import rtc

__version__ = "0.0.0-auto.0"
__repo__ = "https://github.com/adafruit/Adafruit_CircuitPython_NTP.git"


class NTP:
    """Network Time Protocol (NTP) helper module for CircuitPython.
    This module does not handle daylight savings or local time.

    :param adafruit_esp32spi esp: ESP32SPI object.
    """

    def __init__(self, esp):
        # Verify ESP32SPI module
        if "ESP_SPIcontrol" in str(type(esp)):
            self._esp = esp
        else:
            raise TypeError("Provided object is not an ESP_SPIcontrol object.")
        self.valid_time = False

    def set_time(self, tz_offset=0):
        """Fetches and sets the microcontroller's current time
        in seconds since since Jan 1, 1970.

        :param int tz_offset: Timezone offset from GMT
        """
        try:
            now = self._esp.get_time()
            now = time.localtime(now[0] + (tz_offset * 3600))  # 3600 seconds in an hour
            rtc.RTC().datetime = now
            self.valid_time = True
        except ValueError as error:
            print(str(error))
            return
