# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['async_tkinter_loop']
install_requires = \
['asyncio>=3.4.3,<4.0.0']

extras_require = \
{'examples': ['Pillow>=9.1.1,<10.0.0', 'httpx>=0.23.0,<0.24.0']}

setup_kwargs = {
    'name': 'async-tkinter-loop',
    'version': '0.6.0',
    'description': 'Asynchronous mainloop implementation for tkinter',
    'long_description': '# Asynchronous Tkinter Mainloop\n\n[![Python package](https://github.com/insolor/asynctk/actions/workflows/python-package.yml/badge.svg)](https://github.com/insolor/asynctk/actions/workflows/python-package.yml)\n[![Coverage Status](https://coveralls.io/repos/github/insolor/async-tkinter-loop/badge.svg?branch=main)](https://coveralls.io/github/insolor/async-tkinter-loop?branch=main)\n[![Maintainability](https://api.codeclimate.com/v1/badges/2566146b14ef72177613/maintainability)](https://codeclimate.com/github/insolor/async-tkinter-loop/maintainability)\n[![PyPI](https://img.shields.io/pypi/v/async-tkinter-loop)](https://pypi.org/project/async-tkinter-loop/)\n![Supported Python versions](https://img.shields.io/pypi/pyversions/async-tkinter-loop)\n<!--![PyPI - Downloads](https://img.shields.io/pypi/dm/async-tkinter-loop)-->\n\nImplementation of asynchronous `mainloop` for tkinter, the use of which allows using `async` handler functions.\nIt is intended to be as simple to use as possible. No fancy unusual syntax or constructions - just use an alternative\nfunction instead of `root.mainloop()` and wrap asynchronous handlers into a helper function.\n\nBased on ideas from:\n\n* my answer on ru.stackoverflow.com: <https://ru.stackoverflow.com/a/1043146>\n* answer of [Terry Jan Reedy](https://stackoverflow.com/users/722804) on stackoverflow.com:\n  <https://stackoverflow.com/a/47896365>\n* answer of [jfs](https://ru.stackoverflow.com/users/23044) on ru.stackoverflow.com:\n  <https://ru.stackoverflow.com/a/804609>\n\n## Installation\n\n### Release version\n\nInstall the package with the following command:\n\n```\npip install async-tkinter-loop\n```\nor\n```\npip install async-tkinter-loop[examples]\n```\n\n- `[examples]` part is needed to install optional dependencies (such as `aiohttp` and `pillow`) to run some of the\n  examples. If you\'re not going to run examples, remove the `[examples]` part from the command\n- Use `pip3` instead of `pip` on Linux systems to install the package for python3 (not python2)\n- Probably you\'ll want to create a virtual environment for experiments with this library, but this is optional.\n- If you want to try examples, download the entire repository as an archive (green "code" button on\n  [the github page](https://github.com/insolor/async-tkinter-loop) →\n  "Download ZIP"), unpack, run any example (of course, you need to install optional dependencies)\n\n### Development version\n\n1. Install [Poetry](https://python-poetry.org), e.g., with `pip install poetry` (`pip3 install poetry`) command\n   (other possible ways of installation see [here](https://python-poetry.org/docs/#installation))\n2. Download and unpack or clone [the repository](https://github.com/insolor/async-tkinter-loop).\n3. Run the command `poetry install` or `poetry install -E examples` (the later command installs optional dependencies\n   needed to run some examples). This command will create `.venv` directory with a virtual environment and\n   install dependencies into it.\n   - Run any example with `poetry run python examples/sparks.py` (insert a file name of an example).\n   - Or activate the virtual environment with `poetry shell` and run an example with `python examples/sparks.py`\n     command. You can also open the directory with the project in some IDE (e.g., PyCharm or VS Code) \n     and select Python interpreter from the virtual environment as a project interpreter,\n     then run examples directly from the IDE.\n\n## Some examples\n\nBasic example:\n```python\nimport asyncio\nimport tkinter as tk\n\nfrom async_tkinter_loop import async_handler, async_mainloop\n\n\nasync def counter():\n    i = 0\n    while True:\n        i += 1\n        label.config(text=str(i))\n        await asyncio.sleep(1.0)\n\n\nroot = tk.Tk()\n\nlabel = tk.Label(root)\nlabel.pack()\n\ntk.Button(root, text="Start", command=async_handler(counter)).pack()\n\nasync_mainloop(root)\n```\n\nAlso, `async_handler` function can be used as a decorator (but it makes a decorated function syncroneous):\n\n```python\nimport asyncio\nimport tkinter as tk\n\nfrom async_tkinter_loop import async_handler, async_mainloop\n\n\n@async_handler\nasync def counter():\n    i = 0\n    while True:\n        i += 1\n        label.config(text=str(i))\n        await asyncio.sleep(1.0)\n\n\nroot = tk.Tk()\n\nlabel = tk.Label(root)\nlabel.pack()\n\ntk.Button(root, text="Start", command=counter).pack()\n\nasync_mainloop(root)\n```\n\nA more practical example, downloading an image from the Internet with [httpx](https://github.com/encode/httpx)\n(you can use [aiohttp](https://github.com/aio-libs/aiohttp) as well)\nand displaying it in the Tkinter window:\n\n```python\nimport tkinter as tk\nfrom io import BytesIO\n\nimport httpx\nfrom PIL import Image, ImageTk\n\nfrom async_tkinter_loop import async_handler, async_mainloop\n\n\nasync def load_image(url):\n    button.config(state=tk.DISABLED)\n    label.config(text="Loading...", image="")\n\n    async with httpx.AsyncClient() as client:\n        response = await client.get(url, follow_redirects=True)\n        if response.status_code != 200:\n            label.config(text=f"HTTP error {response.status_code}")\n        else:\n            content = response.content\n            pil_image = Image.open(BytesIO(content))\n            image = ImageTk.PhotoImage(pil_image)\n            label.image = image\n            label.config(image=image, text="")\n            button.config(state=tk.NORMAL)\n\n\nurl = "https://picsum.photos/800/640"\n\nroot = tk.Tk()\nroot.geometry("800x640")\n\nbutton = tk.Button(root, text="Load an image", command=async_handler(load_image, url))\nbutton.pack()\n\nlabel = tk.Label(root)\nlabel.pack(expand=1, fill=tk.BOTH)\n\nasync_mainloop(root)\n```\n\nMore examples see in the [`examples`](https://github.com/insolor/async-tkinter-loop/tree/main/examples) directory.\n\n\n## Similar projects\n\n* [Starwort/asynctk](https://github.com/Starwort/asynctk) ([on PyPi](https://pypi.org/project/asynctk/))\n* [gottadiveintopython/asynctkinter](https://github.com/gottadiveintopython/asynctkinter) ([on PyPi](https://pypi.org/project/asynctkinter/))\n* [Lucretiel/tkinter-async](https://github.com/Lucretiel/tkinter-async)\n* [fluentpython/asyncio-tkinter](https://github.com/fluentpython/asyncio-tkinter)\n',
    'author': 'insolor',
    'author_email': 'insolor@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/insolor/async-tkinter-loop',
    'py_modules': modules,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
