"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cdk = require("aws-cdk-lib");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
function GetFakeCertificate(scope, id) {
    return acm.Certificate.fromCertificateArn(scope, id, "arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012");
}
function getTestVpc(stack) {
    return defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            cidr: '172.168.0.0/16',
        },
    });
}
test('Test new load balancer and new lambda function', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing',
        LoadBalancerAttributes: [
            {
                Key: "deletion_protection.enabled",
                Value: "false"
            },
            {
                Key: "access_logs.s3.enabled",
                Value: "true"
            },
            {
                Key: "access_logs.s3.bucket",
                Value: {
                    Ref: "testoneE6ACFBB6"
                }
            },
            {
                Key: "access_logs.s3.prefix",
                Value: ""
            }
        ],
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
});
test('Test new load balancer and new lambda function for HTTP api', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    expect(stack).toCountResources('AWS::Lambda::Function', 1);
});
test('Test existing load balancer and new lambda function', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        existingLoadBalancerObj: existingAlb,
        existingVpc: testExistingVpc,
        publicApi: true,
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    expect(stack).toCountResources('AWS::Lambda::Function', 1);
    expect(stack).toHaveResource("AWS::ElasticLoadBalancingV2::LoadBalancer", {
        Scheme: "internet-facing",
    });
});
test('Test new load balancer and existing lambda function', () => {
    const testFunctionName = 'fixed-name';
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = getTestVpc(stack);
    const lambdaFunction = new lambda.Function(stack, 'existing-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        functionName: testFunctionName,
        vpc: testExistingVpc
    });
    const props = {
        existingLambdaObj: lambdaFunction,
        listenerProps: {
            protocol: 'HTTP'
        },
        publicApi: true,
        existingVpc: testExistingVpc
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        FunctionName: testFunctionName
    });
});
test("Test existing load balancer and existing lambda function", () => {
    const testFunctionName = "fixed-name";
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: "us-east-1" },
    });
    const testExistingVpc = getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, "test-alb", {
        vpc: testExistingVpc,
    });
    const lambdaFunction = new lambda.Function(stack, "existing-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: "index.handler",
        functionName: testFunctionName,
        vpc: testExistingVpc,
    });
    const props = {
        existingLambdaObj: lambdaFunction,
        existingLoadBalancerObj: existingAlb,
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")],
        },
        publicApi: true,
        existingVpc: testExistingVpc,
    };
    new lib_1.AlbToLambda(stack, "test-one", props);
    expect(stack).toHaveResource("AWS::ElasticLoadBalancingV2::LoadBalancer", {
        Scheme: "internal",
    });
    expect(stack).toHaveResource("AWS::ElasticLoadBalancingV2::Listener", {
        Protocol: "HTTP",
    });
    expect(stack).toHaveResource("AWS::ElasticLoadBalancingV2::Listener", {
        Protocol: "HTTPS",
    });
    expect(stack).toHaveResource("AWS::ElasticLoadBalancingV2::ListenerCertificate", {});
    expect(stack).toHaveResource("AWS::ElasticLoadBalancingV2::TargetGroup", {
        TargetType: "lambda",
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test('Test new load balancer and new lambda function', () => {
    const testFunctionName = 'fixed-name';
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            functionName: testFunctionName,
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")],
        },
        publicApi: true,
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        FunctionName: testFunctionName
    });
    expect(stack).toCountResources('AWS::EC2::VPC', 1);
});
test('Test HTTPS adding 2 lambda targets, second with rules', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ],
    });
});
test('Test HTTP adding 2 lambda targets, second with rules', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    expect(stack).not.toHaveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ],
    });
});
test('Test new load balancer and new lambda function', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda',
        Name: 'different-name'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
});
test('Test logging turned off', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true,
        logAlbAccessLogs: false,
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(stack).not.toHaveResource('AWS::S3::Bucket', {});
});
test('Check Properties', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const target = new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(target.loadBalancer);
    expect(target.vpc);
    expect(target.lambdaFunction);
    expect(target.listener);
});
test('Test custom ALB properties', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true,
        loadBalancerProps: {
            loadBalancerName: 'custom-name',
        }
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing',
        Name: 'custom-name',
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
});
test('Test custom logging bucket', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true,
        albLoggingBucketProps: {
            bucketName: 'custom-name',
        }
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing',
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    expect(stack).toHaveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        BucketName: 'custom-name'
    });
});
test('Test providing certificateArns is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificateArns: ['	arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012']
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true,
        albLoggingBucketProps: {
            bucketName: 'should-fail'
        }
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    expect(app).toThrowError(/certificateArns is deprecated. Please supply certificates using props.listenerProps.certificates/);
});
test('Test logging off with logBucketProperties provided is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true,
        logAlbAccessLogs: false,
        albLoggingBucketProps: {
            bucketName: 'should-fail'
        }
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    expect(app).toThrowError(/Error - If logAlbAccessLogs is false, supplying albLoggingBucketProps is invalid./);
});
test('Test certificate with HTTP is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")],
            protocol: 'HTTP',
        },
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    expect(app).toThrowError(/HTTP listeners cannot use a certificate/);
});
test('Test new ALB with no listenerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP/);
});
test('Test existing ALB with no listener with no listenerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        existingLoadBalancerObj: existingAlb,
        existingVpc: testExistingVpc,
        publicApi: true,
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP/);
});
test('Test existing ALB with a listener with listenerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
        listenerProps: {
            protocol: 'HTTPS'
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    };
    // Assertion
    expect(app).toThrowError(/This load balancer already has a listener, listenerProps may not be specified/);
});
test('Test second target with no rules is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [GetFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    };
    // Assertion
    expect(app).toThrowError(/When adding a second target to an existing listener, there must be rules provided/);
});
test('Test no cert for an HTTPS listener is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {},
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/A listener using HTTPS protocol requires a certificate/);
});
test('Test existingLoadBalancerObj and loadBalancerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        existingLoadBalancerObj: existingAlb,
        existingVpc: testExistingVpc,
        publicApi: true,
        loadBalancerProps: {
            internetFacing: true
        }
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide loadBalancerProps or existingLoadBalancerObj, but not both./);
});
test('Test existingLoadBalancerObj and no existingVpc is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        existingLoadBalancerObj: existingAlb,
        publicApi: true,
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/An existing ALB already exists in a VPC, that VPC must be provided in props.existingVpc for the rest of the construct to use./);
});
//# sourceMappingURL=data:application/json;base64,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