"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoEventSource = void 0;
const core_1 = require("@aws-cdk/core");
const stream_1 = require("./stream");
/**
 * Use an Amazon DynamoDB stream as an event source for AWS Lambda.
 *
 * @stability stable
 */
class DynamoEventSource extends stream_1.StreamEventSource {
    /**
     * @stability stable
     */
    constructor(table, props) {
        super(props);
        this.table = table;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 1000)) {
            throw new Error(`Maximum batch size must be between 1 and 1000 inclusive (given ${this.props.batchSize})`);
        }
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        if (!this.table.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.table.node.path}`);
        }
        const eventSourceMapping = target.addEventSourceMapping(`DynamoDBEventSource:${core_1.Names.nodeUniqueId(this.table.node)}`, this.enrichMappingOptions({ eventSourceArn: this.table.tableStreamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.table.grantStreamRead(target);
    }
    /**
     * The identifier for this EventSourceMapping.
     *
     * @stability stable
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('DynamoEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.DynamoEventSource = DynamoEventSource;
//# sourceMappingURL=data:application/json;base64,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