"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisEventSource = void 0;
const cdk = require("@aws-cdk/core");
const stream_1 = require("./stream");
/**
 * Use an Amazon Kinesis stream as an event source for AWS Lambda.
 *
 * @stability stable
 */
class KinesisEventSource extends stream_1.StreamEventSource {
    /**
     * @stability stable
     */
    constructor(stream, props) {
        super(props);
        this.stream = stream;
        this._eventSourceMappingId = undefined;
        this.props.batchSize !== undefined && cdk.withResolved(this.props.batchSize, batchSize => {
            if (batchSize < 1 || batchSize > 10000) {
                throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize})`);
            }
        });
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`KinesisEventSource:${cdk.Names.nodeUniqueId(this.stream.node)}`, this.enrichMappingOptions({ eventSourceArn: this.stream.streamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.stream.grantRead(target);
        // The `grantRead` API provides all the permissions recommended by the Kinesis team for reading a stream.
        // `DescribeStream` permissions are not required to read a stream as it's covered by the `DescribeStreamSummary`
        // and `SubscribeToShard` APIs.
        // The Lambda::EventSourceMapping resource validates against the `DescribeStream` permission. So we add it explicitly.
        // FIXME This permission can be removed when the event source mapping resource drops it from validation.
        this.stream.grant(target, 'kinesis:DescribeStream');
    }
    /**
     * The identifier for this EventSourceMapping.
     *
     * @stability stable
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KinesisEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.KinesisEventSource = KinesisEventSource;
//# sourceMappingURL=data:application/json;base64,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