"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SelfManagedKafkaEventSource = exports.ManagedKafkaEventSource = exports.AuthenticationMethod = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const stream_1 = require("./stream");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * The authentication method to use with SelfManagedKafkaEventSource
 */
var AuthenticationMethod;
(function (AuthenticationMethod) {
    /**
     * SASL_SCRAM_512_AUTH authentication method for your Kafka cluster
     */
    AuthenticationMethod["SASL_SCRAM_512_AUTH"] = "SASL_SCRAM_512_AUTH";
    /**
     * SASL_SCRAM_256_AUTH authentication method for your Kafka cluster
     */
    AuthenticationMethod["SASL_SCRAM_256_AUTH"] = "SASL_SCRAM_256_AUTH";
    /**
     * BASIC_AUTH (SASL/PLAIN) authentication method for your Kafka cluster
     */
    AuthenticationMethod["BASIC_AUTH"] = "BASIC_AUTH";
    /**
     * CLIENT_CERTIFICATE_TLS_AUTH (mTLS) authentication method for your Kafka cluster
     */
    AuthenticationMethod["CLIENT_CERTIFICATE_TLS_AUTH"] = "CLIENT_CERTIFICATE_TLS_AUTH";
})(AuthenticationMethod = exports.AuthenticationMethod || (exports.AuthenticationMethod = {}));
/**
 * Use a MSK cluster as a streaming source for AWS Lambda
 */
class ManagedKafkaEventSource extends stream_1.StreamEventSource {
    constructor(props) {
        super(props);
        this._eventSourceMappingId = undefined;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_event_sources_ManagedKafkaEventSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ManagedKafkaEventSource);
            }
            throw error;
        }
        this.innerProps = props;
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`KafkaEventSource:${core_1.Names.nodeUniqueId(target.node)}${this.innerProps.topic}`, this.enrichMappingOptions({
            eventSourceArn: this.innerProps.clusterArn,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
            kafkaTopic: this.innerProps.topic,
        }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
        target.addToRolePolicy(new iam.PolicyStatement({
            actions: ['kafka:DescribeCluster', 'kafka:GetBootstrapBrokers', 'kafka:ListScramSecrets'],
            resources: [this.innerProps.clusterArn],
        }));
        target.role?.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaMSKExecutionRole'));
    }
    sourceAccessConfigurations() {
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            // "Amazon MSK only supports SCRAM-SHA-512 authentication." from https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html#msk-password-limitations
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH,
                uri: this.innerProps.secret.secretArn,
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
    /**
    * The identifier for this EventSourceMapping
    */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KafkaEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.ManagedKafkaEventSource = ManagedKafkaEventSource;
_a = JSII_RTTI_SYMBOL_1;
ManagedKafkaEventSource[_a] = { fqn: "@aws-cdk/aws-lambda-event-sources.ManagedKafkaEventSource", version: "1.195.0" };
/**
 * Use a self hosted Kafka installation as a streaming source for AWS Lambda.
 */
class SelfManagedKafkaEventSource extends stream_1.StreamEventSource {
    constructor(props) {
        super(props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_event_sources_SelfManagedKafkaEventSourceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SelfManagedKafkaEventSource);
            }
            throw error;
        }
        if (props.vpc) {
            if (!props.securityGroup) {
                throw new Error('securityGroup must be set when providing vpc');
            }
            if (!props.vpcSubnets) {
                throw new Error('vpcSubnets must be set when providing vpc');
            }
        }
        else if (!props.secret) {
            throw new Error('secret must be set if Kafka brokers accessed over Internet');
        }
        this.innerProps = props;
    }
    bind(target) {
        if (!core_2.Construct.isConstruct(target)) {
            throw new Error('Function is not a construct. Unexpected error.');
        }
        target.addEventSourceMapping(this.mappingId(target), this.enrichMappingOptions({
            kafkaBootstrapServers: this.innerProps.bootstrapServers,
            kafkaTopic: this.innerProps.topic,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
        }));
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
    }
    mappingId(target) {
        let hash = crypto.createHash('md5');
        hash.update(JSON.stringify(core_1.Stack.of(target).resolve(this.innerProps.bootstrapServers)));
        const idHash = hash.digest('hex');
        return `KafkaEventSource:${idHash}:${this.innerProps.topic}`;
    }
    sourceAccessConfigurations() {
        let authType;
        switch (this.innerProps.authenticationMethod) {
            case AuthenticationMethod.BASIC_AUTH:
                authType = lambda.SourceAccessConfigurationType.BASIC_AUTH;
                break;
            case AuthenticationMethod.CLIENT_CERTIFICATE_TLS_AUTH:
                authType = lambda.SourceAccessConfigurationType.CLIENT_CERTIFICATE_TLS_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_256_AUTH:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_256_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_512_AUTH:
            default:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH;
                break;
        }
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            sourceAccessConfigurations.push({ type: authType, uri: this.innerProps.secret.secretArn });
        }
        if (this.innerProps.vpcSubnets !== undefined && this.innerProps.securityGroup !== undefined) {
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.VPC_SECURITY_GROUP,
                uri: this.innerProps.securityGroup.securityGroupId,
            });
            this.innerProps.vpc?.selectSubnets(this.innerProps.vpcSubnets).subnetIds.forEach((id) => {
                sourceAccessConfigurations.push({ type: lambda.SourceAccessConfigurationType.VPC_SUBNET, uri: id });
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
}
exports.SelfManagedKafkaEventSource = SelfManagedKafkaEventSource;
_b = JSII_RTTI_SYMBOL_1;
SelfManagedKafkaEventSource[_b] = { fqn: "@aws-cdk/aws-lambda-event-sources.SelfManagedKafkaEventSource", version: "1.195.0" };
//# sourceMappingURL=data:application/json;base64,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