"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayRestApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const open_api_gateway_web_acl_1 = require("./waf/open-api-gateway-web-acl");
/**
 * A construct for creating an api gateway rest api based on the definition in the OpenAPI spec.
 */
class OpenApiGatewayRestApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, spec, specPath, operationLookup, defaultAuthorizer, corsOptions, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKeyPrefix = `${inputSpecAsset.s3ObjectKey}-prepared`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        const prepareSpecLambdaName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}PrepareSpec`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                // The output file will include a hash of the prepared spec, which is not known until deploy time since
                                // tokens must be resolved
                                inputSpecAsset.bucket.arnForObjects(`${preparedSpecOutputKeyPrefix}/*`),
                            ],
                        }),
                    ],
                }),
            },
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                        },
                    ],
                },
            ], true);
        });
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpec", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../../lib/construct/prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${prepareSpecLambdaName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                },
            ], true);
        });
        ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
                {
                    id: RuleId,
                    reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
                },
            ], true);
        });
        const prepareSpecOptions = {
            defaultAuthorizerReference: api_gateway_auth_1.serializeAsAuthorizerReference(defaultAuthorizer),
            integrations: Object.fromEntries(Object.entries(integrations).map(([operationId, { authorizer, integration }]) => [
                operationId,
                {
                    integration: integration.render({
                        operationId,
                        scope: this,
                        ...operationLookup[operationId],
                    }),
                    methodAuthorizer: api_gateway_auth_1.serializeAsAuthorizerReference(authorizer),
                },
            ])),
            securitySchemes: api_gateway_auth_1.prepareSecuritySchemes(this, integrations, defaultAuthorizer),
            corsOptions: corsOptions && {
                allowHeaders: corsOptions.allowHeaders || aws_apigateway_1.Cors.DEFAULT_HEADERS,
                allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
                allowOrigins: corsOptions.allowOrigins,
                statusCode: corsOptions.statusCode || 204,
            },
            operationLookup,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        const preparedSpec = prepare_spec_1.prepareApiSpec(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: preparedSpecOutputKeyPrefix,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecCustomResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: aws_apigateway_1.ApiDefinition.fromBucket(inputSpecAsset.bucket, prepareSpecCustomResource.getAttString("outputSpecKey")),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(scope, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // While the api will be updated when the output path from the custom resource changes, CDK still needs to know when
        // to redeploy the api. This is achieved by including a hash of the spec in the logical id (internalised in the
        // addToLogicalId method since this is how changes of individual resources/methods etc trigger redeployments in CDK)
        this.api.latestDeployment?.addToLogicalId(preparedSpec);
        // Grant API Gateway permission to invoke the integrations
        Object.keys(integrations).forEach((operationId) => {
            integrations[operationId].integration.grant({
                operationId,
                scope: this,
                api: this.api,
                ...operationLookup[operationId],
            });
        });
        // Grant API Gateway permission to invoke each custom authorizer lambda (if any)
        api_gateway_integrations_1.getAuthorizerFunctions(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        // Create and associate the web acl if not disabled
        if (!props.webAclOptions?.disable) {
            const acl = new open_api_gateway_web_acl_1.OpenApiGatewayWebAcl(this, `${id}-Acl`, {
                ...props.webAclOptions,
                apiDeploymentStageArn: this.api.deploymentStage.stageArn,
            });
            this.webAcl = acl.webAcl;
            this.ipSet = acl.ipSet;
            this.webAclAssociation = acl.webAclAssociation;
        }
        ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                    appliesTo: [
                        {
                            regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                        },
                    ],
                },
            ], true);
        });
        ["AwsSolutions-APIG2", "AwsPrototyping-APIGWRequestValidation"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "This construct implements fine grained validation via OpenApi.",
                },
            ], true);
        });
    }
}
exports.OpenApiGatewayRestApi = OpenApiGatewayRestApi;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayRestApi[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayRestApi", version: "0.12.21" };
//# sourceMappingURL=data:application/json;base64,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