"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpHeaders = void 0;
const path = require("path");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const edge_function_1 = require("./edge-function");
const with_configuration_1 = require("./with-configuration");
// Blacklisted headers aren't exposed and can't be added by Lambda@Edge functions:
// https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/lambda-requirements-limits.html#lambda-blacklisted-headers
const BLACKLISTED_HEADERS = [
    /^connection$/i,
    /^expect$/i,
    /^keep-alive$/i,
    /^proxy-authenticate$/i,
    /^proxy-authorization$/i,
    /^proxy-connection$/i,
    /^trailer$/i,
    /^upgrade$/i,
    /^x-accel-buffering$/i,
    /^x-accel-charset$/i,
    /^x-accel-limit-rate$/i,
    /^x-accel-redirect$/i,
    /^X-Amz-Cf-.*/i,
    /^X-Amzn-.*/i,
    /^X-Cache.*/i,
    /^X-Edge-.*/i,
    /^X-Forwarded-Proto.*/i,
    /^X-Real-IP$/i,
];
class HttpHeaders extends edge_function_1.EdgeFunction {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            name: 'http-headers',
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'lambdas', 'http-headers')),
            eventType: aws_cloudfront_1.LambdaEdgeEventType.ORIGIN_RESPONSE,
            edgeRole: props.edgeRole,
            configuration: {
                logLevel: (_a = props.logLevel) !== null && _a !== void 0 ? _a : with_configuration_1.LogLevel.WARN,
                httpHeaders: props.httpHeaders,
            },
        });
        Object.keys(props.httpHeaders).forEach((header) => {
            if (BLACKLISTED_HEADERS.some((blheader) => blheader.test(header))) {
                throw new Error(`HttpHeader ${header} is blacklisted and can't be added by Lambda@Edge functions`);
            }
        });
    }
}
exports.HttpHeaders = HttpHeaders;
//# sourceMappingURL=data:application/json;base64,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