# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['gis_metadata', 'gis_metadata.tests']

package_data = \
{'': ['*'], 'gis_metadata.tests': ['data/*']}

install_requires = \
['frozendict>=2.0,<3.0', 'parserutils>=2.0.1,<3.0.0']

setup_kwargs = {
    'name': 'gis-metadata-parser',
    'version': '2.0.1',
    'description': 'Parser for GIS metadata standards including ArcGIS, FGDC and ISO-19115',
    'long_description': '# gis-metadata-parser\n\nXML parsers for GIS metadata that are designed to read in, validate, update and output a core set of properties that have been mapped between the most common standards, currently:\n\n* FGDC\n* ISO-19139 (and ISO-19115)\n* ArcGIS (tested with ArcGIS format 1.0).\n\nThis library is compatible with Python versions 2.7 and 3.4 through 3.6.\n\n[![Build Status](https://api.travis-ci.com/consbio/gis-metadata-parser.png?branch=main)](https://app.travis-ci.com/github/consbio/gis-metadata-parser)\n[![Coverage Status](https://coveralls.io/repos/github/consbio/gis-metadata-parser/badge.svg?branch=main)](https://coveralls.io/github/consbio/gis-metadata-parser?branch=main)\n\n## Installation\nInstall with `pip install gis-metadata-parser`.\n\n## Usage\n\nParsers can be instantiated from files, XML strings or URLs. They can be converted from one standard to another as well.\n```python\nfrom gis_metadata.arcgis_metadata_parser import ArcGISParser\nfrom gis_metadata.fgdc_metadata_parser import FgdcParser\nfrom gis_metadata.iso_metadata_parser import IsoParser\nfrom gis_metadata.metadata_parser import get_metadata_parser\n\n# From file objects\nwith open(r\'/path/to/metadata.xml\') as metadata:\n    fgdc_from_file = FgdcParser(metadata)\n\nwith open(r\'/path/to/metadata.xml\') as metadata:\n    iso_from_file = IsoParser(metadata)\n\n# Detect standard based on root element, metadata\nfgdc_from_string = get_metadata_parser(\n    """\n    <?xml version=\'1.0\' encoding=\'UTF-8\'?>\n    <metadata>\n        <idinfo>\n        </idinfo>\n    </metadata>\n    """\n)\n\n# Detect ArcGIS standard based on root element and its nodes\niso_from_string = get_metadata_parser(\n    """\n    <?xml version=\'1.0\' encoding=\'UTF-8\'?>\n    <metadata>\n        <dataIdInfo/></dataIdInfo>\n        <distInfo/></distInfo>\n        <dqInfo/></dqInfo>\n    </metadata>\n    """\n)\n\n# Detect ISO standard based on root element, MD_Metadata or MI_Metadata\niso_from_string = get_metadata_parser(\n    """\n    <?xml version=\'1.0\' encoding=\'UTF-8\'?>\n    <MD_Metadata>\n        <identificationInfo>\n        </identificationInfo>\n    </MD_Metadata>\n    """\n)\n\n# Convert from one standard to another\nfgdc_converted = iso_from_file.convert_to(FgdcParser)\niso_converted = fgdc_from_file.convert_to(IsoParser)\narcgis_converted = iso_converted.convert_to(ArcGISParser)\n\n# Output supported properties as key value pairs (dict)\nfgdc_key_vals = fgdc_from_file.convert_to(dict)\niso_key_vals = iso_from_file.convert_to(dict)\n```\n\nFinally, the properties of the parser can be updated, validated, applied and output:\n```python\nwith open(r\'/path/to/metadata.xml\') as metadata:\n    fgdc_from_file = FgdcParser(metadata)\n\n# Example simple properties\nfgdc_from_file.title\nfgdc_from_file.abstract\nfgdc_from_file.place_keywords\nfgdc_from_file.thematic_keywords\n\n# :see: gis_metadata.utils.SUPPORTED_PROPS for list of all supported properties\n\n# Complex properties\nfgdc_from_file.attributes\nfgdc_from_file.bounding_box\nfgdc_from_file.contacts\nfgdc_from_file.dates\nfgdc_from_file.digital_forms\nfgdc_from_file.larger_works\nfgdc_from_file.process_steps\nfgdc_from_file.raster_info\n\n# :see: gis_metadata.utils.COMPLEX_DEFINITIONS for structure of all complex properties\n\n# Update properties\nfgdc_from_file.title = \'New Title\'\nfgdc_from_file.dates = {\'type\': \'single\' \'values\': \'1/1/2016\'}\n\n# Apply updates\nfgdc_from_file.validate()                                      # Ensure updated properties are valid\nfgdc_from_file.serialize()                                     # Output updated XML as a string\nfgdc_from_file.write()                                         # Output updated XML to existing file\nfgdc_from_file.write(out_file_or_path=\'/path/to/updated.xml\')  # Output updated XML to new file\n```\n\n## Extending and Customizing\n\n### Tips\n\nThere are a few unwritten (until now) rules about the way the metadata parsers are wired to work:\n\n1. Properties are generally defined by XPATH in each `parser._data_map`\n2. Simple parser properties accept only values of `string` and `list`\'s of `string`\'s\n3. XPATH\'s configured in the data map support references to element attributes: `\'path/to/element/@attr\'`\n4. Complex parser properties are defined by custom parser/updater functions instead of by XPATH\n5. Complex parser properties accept values of type `dict` containing simple properties, or a list of said `dict`\'s\n6. XPATH keys in the data map with leading underscores are parsed, but not validated or written out\n7. XPATH keys in the data map that "shadow" other properties but with a leading underscore serve as secondary values\n8. Secondary values are used in the absence of a primary value if primary location (element or attribute) is missing\n9. Additional underscores indicate further locations to check for missing values, i.e. `title`, `_title`, `__title`\n\nSome examples of existing secondary properties are as follows:\n```python\n# In the ArcGIS parser for distribution contact phone:\n\nARCGIS_TAG_FORMATS = frozendict({\n     ...\n    \'dist_phone\': \'distInfo/distributor/distorCont/rpCntInfo/cntPhone/voiceNum\',\n    \'_dist_phone\': \'distInfo/distributor/distorCont/rpCntInfo/voiceNum\',  # If not in cntPhone\n    ...\n})\n\n# In the FGDC parser for sub-properties in the contacts definition:\n\nFGDC_DEFINITIONS = dict({k: dict(v) for k, v in iteritems(COMPLEX_DEFINITIONS)})\nFGDC_DEFINITIONS[CONTACTS].update({\n    \'_name\': \'{_name}\',\n    \'_organization\': \'{_organization}\'\n})\n...\nclass FgdcParser(MetadataParser):\n    ...\n    def _init_data_map(self):\n        ...\n        ct_format = FGDC_TAG_FORMATS[CONTACTS]\n        fgdc_data_structures[CONTACTS] = format_xpaths(\n            ...\n            name=ct_format.format(ct_path=\'cntperp/cntper\'),\n            _name=ct_format.format(ct_path=\'cntorgp/cntper\'),  # If not in cntperp\n            organization=ct_format.format(ct_path=\'cntperp/cntorg\'),\n            _organization=ct_format.format(ct_path=\'cntorgp/cntorg\'),  # If not in cntperp\n        )\n\n# Also see the ISO parser for secondary and tertiary sub-properties in the attributes definition:\n\nISO_DEFINITIONS = dict({k: dict(v) for k, v in iteritems(COMPLEX_DEFINITIONS)})\nISO_DEFINITIONS[ATTRIBUTES].update({\n    \'_definition_source\': \'{_definition_src}\',\n    \'__definition_source\': \'{__definition_src}\',\n    \'___definition_source\': \'{___definition_src}\'\n})\n```\n\n\n### Examples\n\nAny of the supported parsers can be extended to include more of a standard\'s supported data. In this example we\'ll add two new properties to the `IsoParser`:\n\n* `metadata_language`: a simple string field describing the language of the metadata file itself (not the dataset)\n* `metadata_contacts`: a complex structure with contact info leveraging and enhancing the existing contact structure\n\nThis example will cover:\n\n1. Adding a new simple property\n2. Configuring a secondary location for a property value\n3. Referencing an element attribute in an XPATH\n4. Adding a new complex property\n5. Customizing the complex property to include a new sub-property\n\nAlso, this example is specifically covered by unit tests.\n\n```python\nfrom gis_metadata.iso_metadata_parser import IsoParser\nfrom gis_metadata.utils import COMPLEX_DEFINITIONS, CONTACTS, format_xpaths, ParserProperty\n\n\nclass CustomIsoParser(IsoParser):\n\n    def _init_data_map(self):\n        super(CustomIsoParser, self)._init_data_map()\n\n        # 1. Basic property: text or list (with secondary location referencing `codeListValue` attribute)\n\n        lang_prop = \'metadata_language\'\n        self._data_map[lang_prop] = \'language/CharacterString\'                    # Parse from here if present\n        self._data_map[\'_\' + lang_prop] = \'language/LanguageCode/@codeListValue\'  # Otherwise, try from here\n\n        # 2. Complex structure (reuse of contacts structure plus phone)\n\n        # 2.1 Define some basic variables\n        ct_prop = \'metadata_contacts\'\n        ct_xpath = \'contact/CI_ResponsibleParty/{ct_path}\'\n        ct_defintion = COMPLEX_DEFINITIONS[CONTACTS]\n        ct_defintion[\'phone\'] = \'{phone}\'\n\n        # 2.2 Reuse CONTACT structure to specify locations per prop (adapted from parent to add `phone`)\n        self._data_structures[ct_prop] = format_xpaths(\n            ct_defintion,\n            name=ct_xpath.format(ct_path=\'individualName/CharacterString\'),\n            organization=ct_xpath.format(ct_path=\'organisationName/CharacterString\'),\n            position=ct_xpath.format(ct_path=\'positionName/CharacterString\'),\n            phone=ct_xpath.format(\n                ct_path=\'contactInfo/CI_Contact/phone/CI_Telephone/voice/CharacterString\'\n            ),\n            email=ct_xpath.format(\n                ct_path=\'contactInfo/CI_Contact/address/CI_Address/electronicMailAddress/CharacterString\'\n            )\n        )\n\n        # 2.3 Set the contact root to insert new elements at "contact" level given the defined path:\n        #   \'contact/CI_ResponsibleParty/...\'\n        # By default we would get multiple "CI_ResponsibleParty" elements under a single "contact"\n        # This way we get multiple "contact" elements, each with its own single "CI_ResponsibleParty"\n        self._data_map[\'_{prop}_root\'.format(prop=ct_prop)] = \'contact\'\n\n        # 2.4 Leverage the default methods for parsing complex properties (or write your own parser/updater)\n        self._data_map[ct_prop] = ParserProperty(self._parse_complex_list, self._update_complex_list)\n\n        # 3. And finally, let the parent validation logic know about the two new custom properties\n\n        self._metadata_props.add(lang_prop)\n        self._metadata_props.add(ct_prop)\n\n\nwith open(r\'/path/to/metadata.xml\') as metadata:\n    iso_from_file = CustomIsoParser(metadata)\n\niso_from_file.metadata_language\niso_from_file.metadata_contacts\n```\n',
    'author': 'dharvey-consbio',
    'author_email': 'dani.harvey@consbio.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/consbio/gis-metadata-parser/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
